/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.governance.rest;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.mac.governance.GovernanceDocumentation;
import org.gecko.mac.governance.GovernanceFactory;
import org.gecko.mac.mgmt.governanceapi.GovernanceDocumentationService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.management.ObjectMetadataContainer;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.jakartars.whiteboard.JakartarsWhiteboardConstants;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsApplicationSelect;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsName;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsResource;
import org.osgi.util.promise.Promise;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * REST Resource for Governance Documentation Service operations.
 * 
 * <p>Provides comprehensive REST API for governance documentation management including:</p>
 * <ul>
 * <li>Documentation storage and retrieval with versioning</li>
 * <li>Latest documentation access and history tracking</li>
 * <li>Documentation existence checks and statistics</li>
 * <li>Documentation deletion and cleanup operations</li>
 * </ul>
 * 
 * <h3>API Endpoints:</h3>
 * <ul>
 * <li><strong>POST /governance/documentation/{objectId}</strong> - Store documentation for an object</li>
 * <li><strong>GET /governance/documentation/{objectId}/latest</strong> - Get latest documentation</li>
 * <li><strong>GET /governance/documentation/by-id/{documentationId}</strong> - Get specific documentation by ID</li>
 * <li><strong>GET /governance/documentation/{objectId}/history</strong> - Get documentation history</li>
 * <li><strong>GET /governance/documentation/{objectId}/exists</strong> - Check if documentation exists</li>
 * <li><strong>DELETE /governance/documentation/{objectId}</strong> - Delete all documentation for object</li>
 * <li><strong>GET /governance/documentation/statistics</strong> - Get documentation statistics</li>
 * </ul>
 * 
 * <h3>EMF XML Serialization</h3>
 * <p>All GovernanceDocumentation responses are serialized using EMF's XML support,
 * consistent with the workflow resource pattern. The Jakarta REST infrastructure 
 * automatically handles EMF object serialization via registered MessageBodyWriters.</p>
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
@RequireJakartarsWhiteboard
@JakartarsResource
@JakartarsName("governance-documentation")
@JakartarsApplicationSelect("(" + JakartarsWhiteboardConstants.JAKARTA_RS_NAME + "=governance)")
@Component(service = GovernanceDocumentationResource.class, scope = ServiceScope.PROTOTYPE)
@Path("/governance/documentation")
@Produces("application/xmi")
@Consumes("application/xmi")
public class GovernanceDocumentationResource {

    private static final Logger LOGGER = Logger.getLogger(GovernanceDocumentationResource.class.getName());

    @Reference
    private GovernanceDocumentationService documentationService;
    @Reference
    private GovernanceFactory govFactory;
    @Reference
    private ManagementFactory mgmtFactory;
    
    @Activate
    public void activate() {
        LOGGER.info("GovernanceDocumentationResource ACTIVATED - documentationService: " + (documentationService != null ? "available" : "null") + 
                    ", govFactory: " + (govFactory != null ? "available" : "null") + 
                    ", mgmtFactory: " + (mgmtFactory != null ? "available" : "null"));
    }
    
    @Deactivate
    public void deactivate() {
        LOGGER.info("GovernanceDocumentationResource DEACTIVATED");
    }

    /**
     * Store governance documentation for an object.
     * Creates both versioned and latest documentation entries for comprehensive audit trail.
     * 
     * @param objectId the object ID to associate documentation with
     * @param reviewUser the user performing the review (required query parameter)
     * @param reason the reason for storing the documentation (optional query parameter)
     * @param documentation the governance documentation to store (XML body)
     * @return Response with the stored documentation ID and object details
     */
    @POST
    @Path("/{objectId}")
    public Response storeDocumentation(
            @PathParam("objectId") String objectId,
            @QueryParam("reviewUser") String reviewUser,
            @QueryParam("reason") String reason,
            GovernanceDocumentation documentation) {
        
        try {
            // Validate required parameters
            if (Objects.isNull(objectId) || objectId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Object ID is required")
                        .build();
            }
            
            if (Objects.isNull(reviewUser) || reviewUser.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("reviewUser query parameter is required")
                        .build();
            }
            
            if (Objects.isNull(documentation)) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Documentation content is required in request body")
                        .build();
            }

            // Store documentation using the service
            Promise<String> storePromise = documentationService.storeDocumentation(objectId, documentation, reviewUser, reason);
            String documentationId = getPromiseValue(storePromise);
            
            LOGGER.info("Stored governance documentation: " + documentationId + " for object: " + objectId + " by user: " + reviewUser);
            
            return Response.status(Response.Status.CREATED)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Documentation stored successfully. Object: " + objectId + ", User: " + reviewUser)
                    .build();
                    
        } catch (IllegalArgumentException e) {
            LOGGER.log(Level.WARNING, "Invalid request for storing documentation for object: " + objectId, e);
            return Response.status(Response.Status.BAD_REQUEST)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Invalid request: " + e.getMessage())
                    .build();
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error storing documentation for object: " + objectId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to store documentation: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Retrieve the latest governance documentation for an object.
     * Returns the most recent documentation entry for fast access.
     * 
     * @param objectId the object ID to get documentation for
     * @return Response with the latest GovernanceDocumentation or 404 if not found
     */
    @GET
    @Path("/{objectId}/latest")
    public Response getLatestDocumentation(@PathParam("objectId") String objectId) {
        try {
            if (Objects.isNull(objectId) || objectId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Object ID is required")
                        .build();
            }

            Optional<GovernanceDocumentation> documentation = documentationService.getLatestDocumentation(objectId);
            
            if (documentation.isPresent()) {
                LOGGER.fine("Retrieved latest documentation for object: " + objectId);
                return Response.ok(documentation.get()).type("application/xmi").build();
            } else {
                LOGGER.fine("No documentation found for object: " + objectId);
                return Response.status(Response.Status.NOT_FOUND)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("No documentation found for object: " + objectId)
                        .build();
            }
            
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error retrieving latest documentation for object: " + objectId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to retrieve documentation: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Retrieve specific governance documentation by its unique ID.
     * Use this endpoint when you have a specific documentation version ID.
     * 
     * @param documentationId the documentation ID to retrieve
     * @return Response with the GovernanceDocumentation or 404 if not found
     */
    @GET
    @Path("/by-id/{documentationId}")
    public Response getDocumentation(@PathParam("documentationId") String documentationId) {
        try {
            if (Objects.isNull(documentationId) || documentationId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Documentation ID is required")
                        .build();
            }

            Optional<GovernanceDocumentation> documentation = documentationService.getDocumentation(documentationId);
            
            if (documentation.isPresent()) {
                LOGGER.fine("Retrieved documentation by ID: " + documentationId);
                return Response.ok(documentation.get()).type("application/xmi").build();
            } else {
                LOGGER.fine("Documentation not found for ID: " + documentationId);
                return Response.status(Response.Status.NOT_FOUND)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Documentation not found: " + documentationId)
                        .build();
            }
            
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error retrieving documentation by ID: " + documentationId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to retrieve documentation: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Get the complete governance documentation history for an object.
     * Returns all versioned documentation ordered by creation time for comprehensive audit trail.
     * 
     * @param objectId the object ID to get history for
     * @return Response with the documentation history metadata list
     */
	@GET
    @Path("/{objectId}/history")
    @Produces({MediaType.TEXT_PLAIN, "application/xmi"})
    public Response getDocumentationHistory(@PathParam("objectId") String objectId) {
        try {
            if (Objects.isNull(objectId) || objectId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Object ID is required")
                        .build();
            }

            List<ObjectMetadata> history = documentationService.getDocumentationHistory(objectId);
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            container.getMetadata().addAll(history);
            LOGGER.fine("Retrieved documentation history for object: " + objectId + " (count: " + history.size() + ")");
            
            return Response.ok(container).type("application/xmi").build();
                    
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error retrieving documentation history for object: " + objectId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to retrieve documentation history: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Check if governance documentation exists for an object.
     * Provides a quick existence check without retrieving the full documentation content.
     * 
     * @param objectId the object ID to check
     * @return Response with existence status (true/false)
     */
    @GET
    @Path("/{objectId}/exists")
    @Produces(MediaType.TEXT_PLAIN)
    public Response hasDocumentation(@PathParam("objectId") String objectId) {
        try {
            if (Objects.isNull(objectId) || objectId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Object ID is required")
                        .build();
            }

            Boolean exists = documentationService.hasDocumentation(objectId);
            
            LOGGER.fine("Documentation existence check for object: " + objectId + " = " + exists);
            
            return Response.ok()
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Object: " + objectId + ", Exists: " + exists)
                    .build();
                    
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error checking documentation existence for object: " + objectId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to check documentation existence: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Delete all governance documentation for an object.
     * Removes both versioned and latest documentation entries.
     * 
     * <p><strong>Warning:</strong> This operation destroys the complete compliance audit trail
     * for the object. Use with extreme caution and proper authorization.</p>
     * 
     * @param objectId the object ID to delete documentation for
     * @return Response with deletion result (HTTP 200 for success, 404 if not found)
     */
    @DELETE
    @Path("/{objectId}")
    @Produces(MediaType.TEXT_PLAIN)
    public Response deleteAllDocumentation(@PathParam("objectId") String objectId) {
        try {
            if (Objects.isNull(objectId) || objectId.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("Object ID is required")
                        .build();
            }

            Promise<Boolean> deletePromise = documentationService.deleteAllDocumentation(objectId);
            Boolean deleted = getPromiseValue(deletePromise);
            
            if (deleted) {
                LOGGER.warning("Deleted ALL documentation for object: " + objectId + " - audit trail destroyed!");
                return Response.ok()
                        .type(MediaType.TEXT_PLAIN)
                        .entity("All documentation deleted for object: " + objectId + ". WARNING: Compliance audit trail has been destroyed")
                        .build();
            } else {
                LOGGER.fine("No documentation found to delete for object: " + objectId);
                return Response.status(Response.Status.NOT_FOUND)
                        .type(MediaType.TEXT_PLAIN)
                        .entity("No documentation found to delete for object: " + objectId)
                        .build();
            }
            
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error deleting documentation for object: " + objectId, e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to delete documentation: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Get comprehensive governance documentation statistics.
     * Includes total documentation count, active objects, historical compliance metrics,
     * and other operational insights for monitoring and reporting.
     * 
     * @return Response with detailed documentation statistics map
     */
    @GET
    @Path("/statistics")
    @Produces(MediaType.TEXT_PLAIN)
    public Response getDocumentationStatistics() {
        try {
            Map<String, Object> statistics = documentationService.getDocumentationStatistics();
            
            LOGGER.fine("Retrieved documentation statistics: " + statistics.size() + " metrics");
            
            return Response.ok()
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Documentation statistics retrieved successfully. Statistics: " + statistics.toString())
                    .build();
                    
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error retrieving documentation statistics", e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                    .type(MediaType.TEXT_PLAIN)
                    .entity("Failed to retrieve statistics: " + e.getMessage())
                    .build();
        }
    }

    /**
     * Helper method to unwrap Promise results with proper exception handling.
     * Waits for Promise completion and handles both successful and failed executions.
     * 
     * @param <T> the promise result type
     * @param promise the promise to unwrap
     * @return the promise value
     * @throws RuntimeException if the promise execution fails
     */
    private <T> T getPromiseValue(Promise<T> promise) {
        try {
            return promise.getValue();
        } catch (InvocationTargetException e) {
            // Unwrap the actual cause from InvocationTargetException
            Throwable cause = e.getCause();
            if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else if (cause instanceof Error) {
                throw (Error) cause;
            } else {
                throw new RuntimeException("Promise execution failed", cause);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Promise execution was interrupted", e);
        }
    }
}