/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.governance.rest;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static java.util.Objects.requireNonNull;

import java.time.Instant;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.emf.rest.annotations.RequireEMFMessageBodyReaderWriter;
import org.gecko.mac.governance.GovernanceDocumentation;
import org.gecko.mac.governance.GovernanceFactory;
import org.gecko.mac.governance.PolicyType;
import org.gecko.mac.mgmt.governanceapi.EObjectWorkflowService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.management.ObjectMetadataContainer;
import org.gecko.mac.mgmt.management.ObjectStatus;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.jakartars.whiteboard.JakartarsWhiteboardConstants;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsApplicationSelect;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsName;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsResource;
import org.osgi.util.promise.Promise;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * REST resource for EObject workflow management operations.
 * 
 * <p>This resource provides HTTP endpoints for managing EObjects through the governance workflow,
 * including draft management, compliance checking, approval/rejection, and release operations.
 * All operations are synchronous from the client perspective - we wait for Promise resolution
 * before returning responses.</p>
 * 
 * <h3>Draft Management Endpoints</h3>
 * <ul>
 * <li><strong>POST /workflow/drafts</strong> - Upload new EObject draft with metadata</li>
 * <li><strong>GET /workflow/drafts</strong> - List all draft objects</li>
 * <li><strong>GET /workflow/drafts/{objectId}</strong> - Get draft metadata by ID</li>
 * <li><strong>GET /workflow/drafts/{objectId}/content</strong> - Get draft EObject content</li>
 * <li><strong>PUT /workflow/drafts/{objectId}</strong> - Update existing draft object</li>
 * <li><strong>DELETE /workflow/drafts/{objectId}</strong> - Delete draft object</li>
 * </ul>
 * 
 * <h3>Compliance Management Endpoints</h3>
 * <ul>
 * <li><strong>POST /workflow/drafts/{objectId}/check</strong> - Run compliance checks on draft</li>
 * <li><strong>POST /workflow/objects/{objectId}/compliance</strong> - Run compliance checks with specific policies</li>
 * <li><strong>GET /workflow/objects/{objectId}/compliance</strong> - Get current compliance status</li>
 * </ul>
 * 
 * <h3>Workflow State Management Endpoints</h3>
 * <ul>
 * <li><strong>POST /workflow/objects/{objectId}/approve</strong> - Approve object for release</li>
 * <li><strong>POST /workflow/objects/{objectId}/reject</strong> - Reject object during review</li>
 * <li><strong>POST /workflow/objects/{objectId}/release</strong> - Release approved object to production</li>
 * </ul>
 * 
 * <h3>Governance Documentation State Management Endpoints</h3>
 * <ul>
 * <li><strong>POST /workflow/objects/{objectId}/governance/draft</strong> - Set governance documentation to DRAFT status</li>
 * <li><strong>POST /workflow/objects/{objectId}/governance/in-review</strong> - Set governance documentation to IN_REVIEW status</li>
 * <li><strong>POST /workflow/objects/{objectId}/governance/approve</strong> - Set governance documentation to APPROVED status</li>
 * <li><strong>POST /workflow/objects/{objectId}/governance/reject</strong> - Set governance documentation to REJECTED status</li>
 * </ul>
 * 
 * <h3>Object Listing Endpoints</h3>
 * <ul>
 * <li><strong>GET /workflow/objects/approved</strong> - List all approved objects</li>
 * <li><strong>GET /workflow/objects/rejected</strong> - List all rejected objects</li>
 * <li><strong>GET /workflow/objects/released</strong> - List all released objects</li>
 * <li><strong>GET /workflow/objects/{objectId}</strong> - Get object metadata by ID</li>
 * <li><strong>GET /workflow/objects/{objectId}/content</strong> - Get object content by ID</li>
 * </ul>
 * 
 * <h3>EMF Serialization</h3>
 * <p>All EObject responses are directly serialized using EMF's XML support.
 * The Jakarta REST infrastructure automatically handles EMF object serialization
 * via registered MessageBodyWriters.</p>
 */
@RequireJakartarsWhiteboard
@RequireEMF
@RequireEMFMessageBodyReaderWriter
@JakartarsResource
@JakartarsName("workflow")
@JakartarsApplicationSelect("(" + JakartarsWhiteboardConstants.JAKARTA_RS_NAME + "=governance)")
@Component(service = WorkflowResource.class, scope = ServiceScope.PROTOTYPE)
@Path("/workflow")
@Produces("application/xmi")
@Consumes("application/xmi")
public class WorkflowResource {

    @Reference
    private EObjectWorkflowService<EObject> workflowService;
    
    @Reference
    private GovernanceFactory govFactory;
    @Reference
    private ManagementFactory mgmtFactory;

    // ======================
    // Draft Management APIs
    // ======================

    /**
     * Upload a new EObject draft with metadata for review.
     * 
     * @param eObject the EObject to upload as draft
     * @param uploadUser the user uploading the draft
     * @param sourceChannel the source channel (e.g. AI_GENERATOR, MANUAL_UPLOAD)
     * @param objectType the type of object (e.g. EPackage, Route, SensorModel)
     * @param contentHash optional content hash for deduplication
     * @param fileExtension optional file extension (e.g. .ecore, .json)
     * @return HTTP 201 with the stored object ID, or 400/500 for errors
     */
    @POST
    @Path("/drafts")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response uploadDraft(EPackage ePackage,
                               @QueryParam("uploadUser") String uploadUser,
                               @QueryParam("sourceChannel") String sourceChannel,
                               @QueryParam("objectType") String objectType,
                               @QueryParam("contentHash") String contentHash,
                               @QueryParam("fileExtension") String fileExtension) {
        try {
            // Validate required parameters
            if (isNull(uploadUser) || uploadUser.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("uploadUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            if (isNull(sourceChannel) || sourceChannel.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("sourceChannel query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            if (isNull(objectType) || objectType.trim().isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("objectType query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            // Create metadata object
            ObjectMetadata metadata = ManagementFactory.eINSTANCE.createObjectMetadata();
            metadata.setUploadUser(uploadUser.trim());
            metadata.setSourceChannel(sourceChannel.trim());
            metadata.setObjectType(objectType.trim());
            metadata.setUploadTime(Instant.now());
            metadata.setObjectName(ePackage.getName());
            metadata.setStatus(ObjectStatus.DRAFT);
            
            if (nonNull(contentHash) && !contentHash.trim().isEmpty()) {
                metadata.setContentHash(contentHash.trim());
            }
            
            if (nonNull(fileExtension) && !fileExtension.trim().isEmpty()) {
                String extension = fileExtension.trim();
                if (!extension.startsWith(".")) {
                    extension = "." + extension;
                }
                metadata.getProperties().put("file.extension", extension);
            }
            
            Promise<String> promise = workflowService.uploadDraft(ePackage, metadata);
            String objectId = promise.getValue(); // Wait for completion
            return Response.status(Response.Status.CREATED)
                          .entity(objectId)
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.BAD_REQUEST)
                          .entity(e.getCause().getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Draft upload failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * List all objects in draft/review status.
     * 
     * @return HTTP 200 with list of draft metadata objects
     */
    @GET
    @Path("/drafts")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response listDraftObjects() {
        try {
            List<ObjectMetadata> drafts = workflowService.listDraftObjects();
            requireNonNull(drafts, "listDraftObjects() must never return null");
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            container.getMetadata().addAll(drafts);
            return Response.ok(container).type("application/xmi").build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("List drafts operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Get draft object metadata by ID.
     * 
     * @param objectId the object identifier
     * @return HTTP 200 with metadata, 404 if not found, or 500 for errors
     */
    @GET
    @Path("/drafts/{objectId}")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response getDraft(@PathParam("objectId") String objectId) {
        try {
            ObjectMetadata metadata = workflowService.getDraft(objectId);
            
            if (nonNull(metadata)) {
                return Response.ok(metadata).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Draft not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Get draft operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Get draft EObject content by ID.
     * 
     * @param objectId the object identifier
     * @return HTTP 200 with EObject content, 404 if not found, or 500 for errors
     */
    @GET
    @Path("/drafts/{objectId}/content")
    public Response getDraftContent(@PathParam("objectId") String objectId) {
        try {
            EObject content = workflowService.getDraftContent(objectId);
            
            if (nonNull(content)) {
                return Response.ok(content).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Draft content not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Get draft content operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Update an existing draft object.
     * 
     * @param objectId the object identifier
     * @param updatedObject the updated EObject
     * @return HTTP 200 with success message, 404 if not found, or 500 for errors
     */
    @PUT
    @Path("/drafts/{objectId}")
    public Response updateDraft(@PathParam("objectId") String objectId, EObject updatedObject) {
        try {
            Promise<Void> promise = workflowService.updateDraft(objectId, updatedObject);
            promise.getValue(); // Wait for completion
            
            return Response.ok("Draft updated successfully").type(MediaType.TEXT_PLAIN).build();
            
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.NOT_FOUND)
                          .entity("Draft not found: " + objectId)
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Update draft operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Delete a draft object.
     * 
     * @param objectId the object identifier
     * @return HTTP 204 if deleted, 404 if not found, or 500 for errors
     */
    @DELETE
    @Path("/drafts/{objectId}")
    public Response deleteDraft(@PathParam("objectId") String objectId) {
        try {
            Promise<Boolean> promise = workflowService.deleteDraft(objectId);
            Boolean success = promise.getValue(); // Wait for completion
            
            if (success) {
                return Response.noContent().build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Draft not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Delete draft operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    // ===========================
    // Compliance Management APIs
    // ===========================

    /**
     * Run comprehensive governance compliance checks on a draft object.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the review
     * @param complianceReason reason for the compliance check
     * @return HTTP 200 with GovernanceDocumentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/drafts/{objectId}/check")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response checkDraft(@PathParam("objectId") String objectId,
                              @QueryParam("reviewUser") String reviewUser,
                              @QueryParam("complianceReason") String complianceReason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            Promise<GovernanceDocumentation> promise = workflowService.checkDraft(objectId, reviewUser, complianceReason);
            GovernanceDocumentation documentation = promise.getValue(); // Wait for completion
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Draft not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Compliance check failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Run specific compliance checks on an object for given policy types.
     * 
     * @param objectId the object identifier
     * @param policyTypes list of policy types to check
     * @param reviewUser the user performing the review
     * @return HTTP 200 with GovernanceDocumentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/compliance")
    public Response runComplianceChecks(@PathParam("objectId") String objectId,
                                       List<PolicyType> policyTypes,
                                       @QueryParam("reviewUser") String reviewUser) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            Promise<GovernanceDocumentation> promise = workflowService.runComplianceChecks(objectId, policyTypes, reviewUser);
            GovernanceDocumentation documentation = promise.getValue(); // Wait for completion
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Compliance checks failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Get the current compliance status for an object.
     * 
     * @param objectId the object identifier
     * @return HTTP 200 with GovernanceDocumentation, 404 if not found, or 500 for errors
     */
    @GET
    @Path("/objects/{objectId}/compliance")
    public Response getComplianceStatus(@PathParam("objectId") String objectId) {
        try {
            GovernanceDocumentation documentation = workflowService.getComplianceStatus(objectId);
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Compliance status not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Get compliance status failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    // ==============================
    // Workflow State Management APIs
    // ==============================

    // ====================================
    // Governance Documentation State APIs
    // ====================================

    /**
     * Set governance documentation status to DRAFT with audit trail.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the state transition
     * @param reason reason for setting to DRAFT status
     * @return HTTP 200 with updated governance documentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/governance/draft")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    @Consumes(MediaType.WILDCARD)
    public Response setGovernanceDocumentationDraft(@PathParam("objectId") String objectId,
                                                    @QueryParam("reviewUser") String reviewUser,
                                                    @QueryParam("reason") String reason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            GovernanceDocumentation documentation = workflowService.setGovernanceDocumentationDraft(objectId, reviewUser, reason);
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Set governance documentation to DRAFT failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Set governance documentation status to IN_REVIEW with audit trail.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the state transition
     * @param reason reason for setting to IN_REVIEW status
     * @return HTTP 200 with updated governance documentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/governance/in-review")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    @Consumes(MediaType.WILDCARD)
    public Response setGovernanceDocumentationInReview(@PathParam("objectId") String objectId,
                                                       @QueryParam("reviewUser") String reviewUser,
                                                       @QueryParam("reason") String reason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            GovernanceDocumentation documentation = workflowService.setGovernanceDocumentationInReview(objectId, reviewUser, reason);
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Set governance documentation to IN_REVIEW failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Set governance documentation status to APPROVED with audit trail.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the state transition
     * @param reason reason for approving the governance documentation
     * @return HTTP 200 with updated governance documentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/governance/approve")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    @Consumes(MediaType.WILDCARD)
    public Response setGovernanceDocumentationApproved(@PathParam("objectId") String objectId,
                                                       @QueryParam("reviewUser") String reviewUser,
                                                       @QueryParam("reason") String reason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            GovernanceDocumentation documentation = workflowService.setGovernanceDocumentationApproved(objectId, reviewUser, reason);
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Set governance documentation to APPROVED failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Set governance documentation status to REJECTED with audit trail.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the state transition
     * @param reason reason for rejecting the governance documentation
     * @return HTTP 200 with updated governance documentation, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/governance/reject")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    @Consumes(MediaType.WILDCARD)
    public Response setGovernanceDocumentationRejected(@PathParam("objectId") String objectId,
                                                       @QueryParam("reviewUser") String reviewUser,
                                                       @QueryParam("reason") String reason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            GovernanceDocumentation documentation = workflowService.setGovernanceDocumentationRejected(objectId, reviewUser, reason);
            
            if (nonNull(documentation)) {
                return Response.ok(documentation).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Set governance documentation to REJECTED failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Approve an object for release.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the approval
     * @param approvalReason reason for approval
     * @return HTTP 200 with updated metadata, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/approve")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response approveObject(@PathParam("objectId") String objectId,
                                 @QueryParam("reviewUser") String reviewUser,
                                 @QueryParam("approvalReason") String approvalReason) {
        try {
            if (isNull(reviewUser)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser query parameter is required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            ObjectMetadata updatedMetadata = workflowService.approveObject(objectId, reviewUser, approvalReason);
            
            if (nonNull(updatedMetadata)) {
                return Response.ok(updatedMetadata).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Approval operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Reject an object during review.
     * 
     * @param objectId the object identifier
     * @param reviewUser the user performing the rejection
     * @param rejectionReason reason for rejection
     * @return HTTP 200 with updated metadata, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/reject")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response rejectObject(@PathParam("objectId") String objectId,
                                @QueryParam("reviewUser") String reviewUser,
                                @QueryParam("rejectionReason") String rejectionReason) {
        try {
            if (isNull(reviewUser) || isNull(rejectionReason)) {
                return Response.status(Response.Status.BAD_REQUEST)
                              .entity("reviewUser and rejectionReason query parameters are required")
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
            
            ObjectMetadata updatedMetadata = workflowService.rejectObject(objectId, reviewUser, rejectionReason);
            
            if (nonNull(updatedMetadata)) {
                return Response.ok(updatedMetadata).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Rejection operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Release an approved object to production storage after compliance checks.
     * 
     * @param objectId the object identifier
     * @param releaseNotes notes for the release
     * @param requireComplianceCheck whether to require compliance check before release
     * @return HTTP 200 with updated metadata, 404 if not found, or 500 for errors
     */
    @POST
    @Path("/objects/{objectId}/release")
    public Response releaseObject(@PathParam("objectId") String objectId,
                                 @QueryParam("releaseNotes") String releaseNotes,
                                 @QueryParam("requireComplianceCheck") Boolean requireComplianceCheck) {
        try {
            boolean requireCheck = nonNull(requireComplianceCheck) ? requireComplianceCheck : true;
            
            ObjectMetadata updatedMetadata = workflowService.releaseObject(objectId, releaseNotes, requireCheck);
            
            if (nonNull(updatedMetadata)) {
                return Response.ok(updatedMetadata).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Release operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    // ======================
    // Object Listing APIs
    // ======================

    /**
     * List all approved objects ready for release.
     * 
     * @return HTTP 200 with list of approved metadata objects
     */
    @GET
    @Path("/objects/approved")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response listApprovedObjects() {
        try {
            List<ObjectMetadata> approved = workflowService.listApprovedObjects();
            requireNonNull(approved, "listApprovedObjects() must never return null");
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            container.getMetadata().addAll(approved);
            return Response.ok(container).type("application/xmi").build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("List approved objects failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * List all objects across all workflow states (draft, rejected, approved, released).
     * 
     * @return HTTP 200 with container of all metadata objects
     */
    @GET
    @Path("/objects/all")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response listAllObjects() {
        try {
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            
            // Collect objects from all workflow states with null safety checks
            List<ObjectMetadata> drafts = workflowService.listDraftObjects();
            requireNonNull(drafts, "listDraftObjects() must never return null");
            
            List<ObjectMetadata> rejected = workflowService.listRejectedObjects();
            requireNonNull(rejected, "listRejectedObjects() must never return null");
            
            List<ObjectMetadata> approved = workflowService.listApprovedObjects();
            requireNonNull(approved, "listApprovedObjects() must never return null");
            
            List<ObjectMetadata> released = workflowService.listReleasedObjects();
            requireNonNull(released, "listReleasedObjects() must never return null");
            
            // Add all objects to the container
            container.getMetadata().addAll(drafts);
            container.getMetadata().addAll(rejected);
            container.getMetadata().addAll(approved);
            container.getMetadata().addAll(released);
            
            return Response.ok(container).type("application/xmi").build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("List all objects failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * List all rejected objects.
     * 
     * @return HTTP 200 with list of rejected metadata objects
     */
    @GET
    @Path("/objects/rejected")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response listRejectedObjects() {
        try {
            List<ObjectMetadata> rejected = workflowService.listRejectedObjects();
            requireNonNull(rejected, "listRejectedObjects() must never return null");
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            container.getMetadata().addAll(rejected);
            return Response.ok(container).type("application/xmi").build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("List rejected objects failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * List all released/production objects.
     * 
     * @return HTTP 200 with list of released metadata objects
     */
    @GET
    @Path("/objects/released")
    @Produces({"application/xmi", MediaType.TEXT_PLAIN})
    public Response listReleasedObjects() {
        try {
            List<ObjectMetadata> released = workflowService.listReleasedObjects();
            requireNonNull(released, "listReleasedObjects() must never return null");
            ObjectMetadataContainer container = mgmtFactory.createObjectMetadataContainer();
            container.getMetadata().addAll(released);
            return Response.ok(container).type("application/xmi").build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("List released objects failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Get object metadata by ID.
     * 
     * @param objectId the object identifier
     * @return HTTP 200 with metadata, 404 if not found, or 500 for errors
     */
    @GET
    @Path("/objects/{objectId}")
    public Response getObject(@PathParam("objectId") String objectId) {
        try {
            ObjectMetadata metadata = workflowService.getObject(objectId);
            
            if (nonNull(metadata)) {
                return Response.ok(metadata).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Get object operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Get object content by ID.
     * 
     * @param objectId the object identifier
     * @return HTTP 200 with EObject content, 404 if not found, or 500 for errors
     */
    @GET
    @Path("/objects/{objectId}/content")
    public Response getObjectContent(@PathParam("objectId") String objectId) {
        try {
            Object content = workflowService.getObjectContent(objectId);
            
            if (nonNull(content)) {
                return Response.ok(content).type("application/xmi").build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object content not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Get object content operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Update an existing object (works across all workflow states).
     * 
     * @param objectId the object identifier
     * @param updatedObject the updated EObject
     * @return HTTP 200 with success message, 404 if not found, or 500 for errors
     */
    @PUT
    @Path("/objects/{objectId}")
    public Response updateObject(@PathParam("objectId") String objectId, EObject updatedObject) {
        try {
            Promise<Void> promise = workflowService.updateObject(objectId, updatedObject);
            promise.getValue(); // Wait for completion
            
            return Response.ok("Object updated successfully")
                          .type(MediaType.TEXT_PLAIN)
                          .build();
            
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.NOT_FOUND)
                          .entity("Object not found: " + objectId)
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Update object operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }

    /**
     * Delete an object (works across all workflow states).
     * 
     * @param objectId the object identifier
     * @return HTTP 204 if deleted, 404 if not found, or 500 for errors
     */
    @DELETE
    @Path("/objects/{objectId}")
    public Response deleteObject(@PathParam("objectId") String objectId) {
        try {
            Promise<Boolean> promise = workflowService.deleteObject(objectId);
            Boolean success = promise.getValue(); // Wait for completion
            
            if (success) {
                return Response.noContent().build();
            } else {
                return Response.status(Response.Status.NOT_FOUND)
                              .entity("Object not found: " + objectId)
                              .type(MediaType.TEXT_PLAIN)
                              .build();
            }
        } catch (Exception e) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
                          .entity("Delete object operation failed: " + e.getMessage())
                          .type(MediaType.TEXT_PLAIN)
                          .build();
        }
    }
}