/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.initializer;

import java.util.List;

import org.eclipse.emf.ecore.EPackage;
import org.gecko.mac.governance.ApprovalStatus;
import org.gecko.mac.mgmt.management.ObjectStatus;
import org.osgi.util.promise.Promise;

/**
 * Service for initializing governance documentation workflows with demo data
 * or custom scenarios. Supports both batch demo initialization and individual
 * model processing with state validation.
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
public interface GovernanceInitializerService {

    /**
     * Initialize documentation with target states using EPackage instance.
     * 
     * @param ePackage EPackage instance (injected as OSGi service)
     * @param documentationPath Path to governance documentation XMI file (relative to bundle)
     * @param targetObjectStatus Target ObjectStatus (APPROVED/REJECTED)
     * @param targetApprovalStatus Target ApprovalStatus (APPROVED/REJECTED) 
     * @return Promise<String> documentation ID assigned by workflow service
     * @throws IllegalArgumentException if state combination is invalid
     */
    Promise<String> initialize(EPackage ePackage, 
                              String documentationPath,
                              ObjectStatus targetObjectStatus, 
                              ApprovalStatus targetApprovalStatus);
    
    /**
     * Initialize documentation with additional context (review reason, user).
     * 
     * @param ePackage EPackage instance (injected as OSGi service)
     * @param documentationPath Path to governance documentation XMI file
     * @param targetObjectStatus Target ObjectStatus (APPROVED/REJECTED)
     * @param targetApprovalStatus Target ApprovalStatus (APPROVED/REJECTED)
     * @param reviewReason Human-readable reason for the approval/rejection decision
     * @param reviewUser User who performed the review (for audit trail)
     * @return Promise<String> documentation ID assigned by workflow service
     * @throws IllegalArgumentException if state combination is invalid
     */
    Promise<String> initialize(EPackage ePackage,
                              String documentationPath, 
                              ObjectStatus targetObjectStatus,
                              ApprovalStatus targetApprovalStatus,
                              String reviewReason,
                              String reviewUser);
    
    /**
     * Initialize with default demo data from governance folder.
     * Processes all 8 pre-configured documentation files with their
     * target states (3 approved, 5 rejected) to demonstrate the
     * complete governance workflow.
     * 
     * @return Promise<List<String>> list of documentation IDs
     */
    Promise<List<String>> initializeDemo();
    
    /**
     * Validate that the given state combination is legal according to
     * the governance workflow rules.
     * 
     * @param objectStatus ObjectStatus to validate
     * @param approvalStatus ApprovalStatus to validate
     * @return true if combination is valid, false otherwise
     */
    boolean isValidStateCombination(ObjectStatus objectStatus, ApprovalStatus approvalStatus);
    
    /**
     * Verify that a completed workflow matches the requested target states.
     * Can be used to validate that the initialization succeeded as expected.
     * 
     * @param documentationId ID returned from initialize()
     * @param expectedObjectStatus Expected final ObjectStatus
     * @param expectedApprovalStatus Expected final ApprovalStatus
     * @return Promise<Boolean> true if states match, false otherwise
     */
    Promise<Boolean> verifyResult(String documentationId,
                                 ObjectStatus expectedObjectStatus,
                                 ApprovalStatus expectedApprovalStatus);
}