/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.initializer;

import java.util.Objects;
import java.util.Set;

import org.gecko.mac.governance.ApprovalStatus;
import org.gecko.mac.mgmt.management.ObjectStatus;

/**
 * Value class representing a combination of ObjectStatus and ApprovalStatus.
 * Provides validation logic for legal state combinations according to the
 * governance workflow rules.
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
public record StateCombination(ObjectStatus objectStatus, ApprovalStatus approvalStatus) {

    // Valid state combinations according to governance workflow
    private static final Set<StateCombination> VALID_COMBINATIONS = Set.of(
        new StateCombination(ObjectStatus.DRAFT, ApprovalStatus.DRAFT),
        new StateCombination(ObjectStatus.DRAFT, ApprovalStatus.IN_REVIEW), 
        new StateCombination(ObjectStatus.APPROVED, ApprovalStatus.APPROVED),
        new StateCombination(ObjectStatus.REJECTED, ApprovalStatus.REJECTED),
        new StateCombination(ObjectStatus.DEPLOYED, ApprovalStatus.APPROVED),
        new StateCombination(ObjectStatus.ARCHIVED, ApprovalStatus.DRAFT),
        new StateCombination(ObjectStatus.ARCHIVED, ApprovalStatus.IN_REVIEW),
        new StateCombination(ObjectStatus.ARCHIVED, ApprovalStatus.APPROVED),
        new StateCombination(ObjectStatus.ARCHIVED, ApprovalStatus.REJECTED)
    );
    
    /**
     * Constructor with validation.
     */
    public StateCombination {
        Objects.requireNonNull(objectStatus, "ObjectStatus cannot be null");
        Objects.requireNonNull(approvalStatus, "ApprovalStatus cannot be null");
    }
    
    /**
     * Check if this state combination is valid according to governance workflow rules.
     * 
     * @return true if combination is valid, false otherwise
     */
    public boolean isValid() {
        return VALID_COMBINATIONS.contains(this);
    }
    
    /**
     * Static factory method for validation.
     * 
     * @param objectStatus ObjectStatus to validate
     * @param approvalStatus ApprovalStatus to validate
     * @return true if combination is valid, false otherwise
     */
    public static boolean isValidCombination(ObjectStatus objectStatus, ApprovalStatus approvalStatus) {
        if (objectStatus == null || approvalStatus == null) {
            return false;
        }
        return new StateCombination(objectStatus, approvalStatus).isValid();
    }
    
    /**
     * Get all valid state combinations for reference.
     * 
     * @return immutable set of valid combinations
     */
    public static Set<StateCombination> getValidCombinations() {
        return VALID_COMBINATIONS;
    }
}