/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.initializer.impl;

import static java.util.Objects.requireNonNull;

import java.net.URL;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.mac.governance.ApprovalStatus;
import org.gecko.mac.governance.GovernanceDocumentation;
import org.gecko.mac.initializer.GovernanceInitializerService;
import org.gecko.mac.initializer.StateCombination;
import org.gecko.mac.mgmt.api.EObjectStorageService;
import org.gecko.mac.mgmt.governanceapi.GovernanceDocumentationService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.management.ObjectStatus;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.Promises;

/**
 * Implementation of the GovernanceInitializerService that provides both demo
 * initialization and custom model processing capabilities.
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
@Component(name = "GovernanceInitializerComponent", immediate = true, configurationPolicy = ConfigurationPolicy.REQUIRE)
public class GovernanceInitializerComponent implements GovernanceInitializerService {

    private static final Logger logger = Logger.getLogger(GovernanceInitializerComponent.class.getName());

    @Reference
    private GovernanceDocumentationService documentationService;
    
    @Reference
    private EObjectStorageService<EObject> draftStorage;
    @Reference
    private EObjectStorageService<EObject> releaseStorage;
    
    @Reference
    private ResourceSet resourceSet;
    
    // Inject all required EPackages as OSGi services
    @Reference(target = "(emf.name=booking)")
    private EPackage golfBookingPackage;
    
    @Reference(target = "(emf.name=equipment)")
    private EPackage golfEquipmentPackage;
    
    @Reference(target = "(emf.name=lorawan)")
    private EPackage lorawanPackage;
    
//    @Reference(target = "(emf.nsURI=https://gecko.org/airquality/1.0)")
//    private EPackage airqualityPackage;
    
    @Reference(target = "(emf.name=membership)")
    private EPackage golfMembershipPackage;
    
    @Reference(target = "(emf.name=player)")
    private EPackage playerPackage;
    
    @Reference(target = "(emf.name=rental)")
    private EPackage golfRentalPackage;
    
    @Reference(target = "(emf.name=transaction)")
    private EPackage golfTransactionPackage;

    @Activate
    public void activate() {
        logger.info("Governance Initializer activated - all required EPackages available");
        logger.info("Available packages: " + 
            List.of(golfBookingPackage.getNsURI(), golfEquipmentPackage.getNsURI(), 
                   lorawanPackage.getNsURI(), golfMembershipPackage.getNsURI(), playerPackage.getNsURI(),
                   golfRentalPackage.getNsURI(), golfTransactionPackage.getNsURI()));
        
        // Initialize demo packages automatically on activation
        logger.info("Starting automatic demo initialization...");
        initializeDemo()
            .then(objectIdsP -> {
                List<String> objectIds = objectIdsP.getValue();
                logger.info("Successfully initialized " + objectIds.size() + " governance documentation packages");
                return Promises.resolved(null);
            });
    }

    @Override
    public Promise<String> initialize(EPackage ePackage, String documentationPath,
                                     ObjectStatus targetObjectStatus, ApprovalStatus targetApprovalStatus) {
        return initialize(ePackage, documentationPath, targetObjectStatus, targetApprovalStatus, null, null);
    }

    @Override
    public Promise<String> initialize(EPackage ePackage, String documentationPath,
                                     ObjectStatus targetObjectStatus, ApprovalStatus targetApprovalStatus,
                                     String reviewReason, String reviewUser) {
        
        // Validate inputs
        requireNonNull(ePackage, "EPackage cannot be null");
        requireNonNull(documentationPath, "Documentation path cannot be null");
        requireNonNull(targetObjectStatus, "Target object status cannot be null");
        requireNonNull(targetApprovalStatus, "Target approval status cannot be null");
        
        // Validate state combination
//        if (!isValidStateCombination(targetObjectStatus, targetApprovalStatus)) {
//            return Promises.failed(new IllegalArgumentException(
//                "Invalid state combination: " + targetObjectStatus + "/" + targetApprovalStatus));
//        }
        
        logger.info("Initializing documentation for EPackage: " + ePackage.getNsURI() + 
                   " -> " + targetObjectStatus + "/" + targetApprovalStatus);
        
        try {
            // Load documentation from bundle resource
            GovernanceDocumentation documentation = loadDocumentation(documentationPath);
            
            // Validate documentation matches EPackage
            if (!documentation.getModelName().toLowerCase().contains(getModelKeyword(ePackage))) {
                logger.warning("Documentation model name '" + documentation.getModelName() + 
                             "' may not match EPackage '" + ePackage.getName() + "'");
            }
            
            // Process through actual workflow service
            return processWorkflowSteps(documentation, ePackage, targetObjectStatus, 
                                      targetApprovalStatus, reviewReason, reviewUser);
            
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Failed to initialize documentation: " + e.getMessage(), e);
            return Promises.failed(e);
        }
    }
    
    @Override
    public Promise<List<String>> initializeDemo() {
        logger.info("Starting demo initialization for all governance documentation");
        
        List<Promise<String>> promises = new ArrayList<>();
        
        // REJECTED scenarios
        promises.add(initialize(golfBookingPackage, 
            "governance/Booking-GovernanceDocumentation.xmi",
            ObjectStatus.DRAFT, ApprovalStatus.REJECTED,
            "Privacy concerns with PII tracking", 
            "Privacy Officer"));
            
        promises.add(initialize(golfMembershipPackage,
            "governance/Membership-GovernanceDocumentation.xmi", 
            ObjectStatus.DRAFT, ApprovalStatus.REJECTED,
            "GDPR compliance issues with member data",
            "Compliance Officer"));
            
        promises.add(initialize(playerPackage,
            "governance/Player-GovernanceDocumentation.xmi",
            ObjectStatus.DRAFT, ApprovalStatus.REJECTED,
            "Personal data handling concerns",
            "Data Protection Officer"));
            
        promises.add(initialize(golfRentalPackage,
            "governance/Rental-GovernanceDocumentation.xmi",
            ObjectStatus.DRAFT, ApprovalStatus.REJECTED,
            "Tracking concerns with equipment usage",
            "Privacy Officer"));
            
        promises.add(initialize(golfTransactionPackage,
            "governance/Transaction-GovernanceDocumentation.xmi",
            ObjectStatus.DRAFT, ApprovalStatus.REJECTED,
            "Financial data sensitivity issues",
            "Security Officer"));
        
        // APPROVED scenarios  
        promises.add(initialize(golfEquipmentPackage,
            "governance/Equipment-GovernanceDocumentation.xmi",
            ObjectStatus.APPROVED, ApprovalStatus.APPROVED,
            "Equipment management approved - no privacy issues",
            "Technical Reviewer"));
            
        promises.add(initialize(lorawanPackage,
            "governance/LoRaWAN-GovernanceDocumentation.xmi", 
            ObjectStatus.APPROVED, ApprovalStatus.APPROVED,
            "Network telemetry data approved - no PII",
            "Network Administrator"));
            
        return Promises.all(promises);
    }

    @Override
    public boolean isValidStateCombination(ObjectStatus objectStatus, ApprovalStatus approvalStatus) {
        return StateCombination.isValidCombination(objectStatus, approvalStatus);
    }

    @Override
    public Promise<Boolean> verifyResult(String objectId, ObjectStatus expectedObjectStatus,
                                        ApprovalStatus expectedApprovalStatus) {
        
        requireNonNull(objectId, "Object ID cannot be null");
        requireNonNull(expectedObjectStatus, "Expected object status cannot be null");
        requireNonNull(expectedApprovalStatus, "Expected approval status cannot be null");
        
        logger.info("Verifying result for object ID: " + objectId);
        
        try {
            // Determine which storage to check based on expected status
            EObjectStorageService<EObject> targetStorage = 
                (expectedObjectStatus == ObjectStatus.APPROVED) ? releaseStorage : draftStorage;
            
            // Get object metadata to check ObjectStatus
            return targetStorage.retrieveMetadata(objectId)
            .then(metadataP -> {
                ObjectMetadata metadata = metadataP.getValue();
                if (metadata == null) {
                    logger.warning("No metadata found for object ID: " + objectId);
                    return Promises.resolved(false);
                }
                
                // Check ObjectStatus
                boolean objectStatusMatch = metadata.getStatus() == expectedObjectStatus;
                if (!objectStatusMatch) {
                    logger.warning("ObjectStatus mismatch for " + objectId + 
                                 ": expected " + expectedObjectStatus + ", got " + metadata.getStatus());
                }
                
                // Get governance documentation to check ApprovalStatus
                Optional<GovernanceDocumentation> docOpt = documentationService.getLatestDocumentation(objectId);
                if (docOpt.isEmpty()) {
                    logger.warning("No governance documentation found for ID: " + objectId);
                    return Promises.resolved(false);
                }
                
                GovernanceDocumentation doc = docOpt.get();
                boolean approvalStatusMatch = doc.getStatus() == expectedApprovalStatus;
                if (!approvalStatusMatch) {
                    logger.warning("ApprovalStatus mismatch for " + objectId + 
                                 ": expected " + expectedApprovalStatus + ", got " + doc.getStatus());
                }
                
                boolean allMatch = objectStatusMatch && approvalStatusMatch;
                logger.info("Verification result for " + objectId + ": " + 
                           (allMatch ? "SUCCESS" : "FAILED") + 
                           " (Object: " + metadata.getStatus() + ", Approval: " + doc.getStatus() + ")");
                
                return Promises.resolved(allMatch);
            });
            
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Failed to verify result: " + e.getMessage(), e);
            return Promises.failed(e);
        }
    }
    
    /**
     * Load governance documentation from bundle resource.
     */
    private GovernanceDocumentation loadDocumentation(String path) throws Exception {
        Bundle bundle = FrameworkUtil.getBundle(getClass());
        URL resourceUrl = bundle.getResource(path);
        
        if (resourceUrl == null) {
            throw new IllegalArgumentException("Documentation not found: " + path);
        }
        
        URI uri = URI.createURI(resourceUrl.toString());
        Resource resource = resourceSet.getResource(uri, true);
        
        if (resource == null || resource.getContents().isEmpty()) {
            throw new IllegalArgumentException("Failed to load documentation: " + path);
        }
        
        Object content = resource.getContents().get(0);
        if (!(content instanceof GovernanceDocumentation)) {
            throw new IllegalArgumentException("Resource is not GovernanceDocumentation: " + path);
        }
        
        return (GovernanceDocumentation) content;
    }
    
    /**
     * Extract a keyword from EPackage name for validation.
     */
    private String getModelKeyword(EPackage ePackage) {
        String name = ePackage.getName().toLowerCase();
        if (name.contains("golf")) return "golf";
        if (name.contains("player")) return "player";
        if (name.contains("lorawan")) return "lorawan";
        if (name.contains("airquality")) return "airq";
        return name;
    }
    
    /**
     * Process the workflow steps to reach target states using direct storage services.
     * This approach bypasses workflow constraints and directly places objects in target storage.
     */
    private Promise<String> processWorkflowSteps(GovernanceDocumentation documentation, EPackage ePackage,
                                                ObjectStatus targetObjectStatus, ApprovalStatus targetApprovalStatus,
                                                String reviewReason, String reviewUser) {
        
        logger.info("Processing direct storage for: " + documentation.getModelName());
        logger.info("Target states: " + targetObjectStatus + "/" + targetApprovalStatus);
        logger.info("Review context: " + reviewUser + " - " + reviewReason);
        
        try {
            // Step 1: Create metadata for the EPackage with target status
            ObjectMetadata metadata = createMetadataForDemo(ePackage, reviewUser);
            metadata.setStatus(targetObjectStatus);
            
            // Step 2: Set documentation approval status
            documentation.setStatus(targetApprovalStatus);
            if (reviewUser != null) {
                metadata.setReviewUser(reviewUser);
            }
            if (reviewReason != null) {
                metadata.setReviewReason(reviewReason);
            }
            
            // Step 3: Choose appropriate storage based on target object status
            EObjectStorageService<EObject> targetStorage = 
                (targetObjectStatus == ObjectStatus.APPROVED) ? releaseStorage : draftStorage;
            
            // Step 4: Store EPackage in target storage
            return targetStorage.storeObject(null, EcoreUtil.copy(ePackage), metadata)
            .then(objectIdP -> {
                String objectId = objectIdP.getValue();
                logger.info("Stored EPackage in " + 
                    (targetStorage == releaseStorage ? "release" : "draft") + 
                    " storage with objectId: " + objectId);
                
                // Step 5: Store governance documentation linked to the object
                return documentationService.storeDocumentation(objectId, documentation, 
                    reviewUser != null ? reviewUser : "System", "Direct initialization")
                .then(docIdP -> {
                    String docId = docIdP.getValue();
                    logger.info("Stored governance documentation " + docId + " for object " + objectId);
                    
                    // Step 6: Link documentation to object metadata
                    metadata.setGovernanceDocumentationId(docId);
                    
                    // Step 7: Update metadata with documentation link
                    return targetStorage.updateMetadata(objectId, metadata)
                    .then(updatedMetadata -> {
                        logger.info("Updated metadata with governance documentation link");
                        return Promises.resolved(objectId);
                    });
                });
            });
            
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Direct storage processing failed: " + e.getMessage(), e);
            return Promises.failed(e);
        }
    }
    
    
    /**
     * Create metadata for demo EPackage objects.
     */
    private ObjectMetadata createMetadataForDemo(EPackage ePackage, String reviewUser) {
        ManagementFactory factory = ManagementFactory.eINSTANCE;
        ObjectMetadata metadata = factory.createObjectMetadata();
        
        metadata.setObjectName("EPackage: " + ePackage.getName());
        metadata.setVersion("1.0.0");
        metadata.setObjectType("EPackage");
        metadata.setUploadUser(reviewUser != null ? reviewUser : "System");
        metadata.setUploadTime(Instant.now());
        metadata.setSourceChannel("GOVERNANCE_INITIALIZER");
        metadata.setLastChangeUser(reviewUser != null ? reviewUser : "System");
        metadata.setLastChangeTime(Instant.now());
        
        // Add EPackage-specific properties
        metadata.getProperties().put("nsURI", ePackage.getNsURI());
        metadata.getProperties().put("nsPrefix", ePackage.getNsPrefix());
        metadata.getProperties().put("packageName", ePackage.getName());
        metadata.getProperties().put("classCount", ePackage.getEClassifiers().size());
        
        return metadata;
    }
}