/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Guido Grune - initial implementation
 */
package org.gecko.mac.logging;

import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.slf4j.bridge.SLF4JBridgeHandler;

/**
 * OSGi component that initializes the JUL (java.util.logging) to SLF4J bridge.
 * This ensures all JUL log statements are routed through SLF4J and ultimately to Logback.
 *
 * <p>The component is activated immediately at bundle start to ensure early logging capture.</p>
 *
 * @since 1.0
 */
@Component
public class JULBridgeInitializer {

    /**
     * Activates the JUL to SLF4J bridge.
     *
     * <p>This method:</p>
     * <ul>
     *   <li>Removes all existing JUL handlers</li>
     *   <li>Installs the SLF4JBridgeHandler to route JUL logs to SLF4J</li>
     *   <li>Sets the root logger level to ALL to capture all log levels</li>
     * </ul>
     */
    @Activate
    public void activate() {
        LogManager.getLogManager().reset();
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
        Logger.getLogger("").setLevel(Level.ALL);
		System.out.println("Logging bridge install (JUL → SLF4J)");
    }

    /**
     * Deactivates the JUL to SLF4J bridge when the component is stopped.
     */
    @Deactivate
    public void deactivate() {
        SLF4JBridgeHandler.uninstall();
		System.out.println("Logging bridge uninstall (JUL → SLF4J)");
    }
}
