/**
 * Copyright (c) 2012 - 2023 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.management.ai.test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertInstanceOf;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.List;

import org.eclipse.emf.ecore.EPackage;
import org.gecko.mac.mgmt.api.EObjectDiscoveryService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;
import org.osgi.util.promise.Promise;

/**
 * Comprehensive test suite for EPackageAIDiscoveryService.
 *
 * Tests cover:
 * - JSON fingerprint generation
 * - Finding objects by JSON pattern
 * - Searching for similar objects
 * - Generation request caching
 * - Object registration caching
 *
 * See documentation here:
 * 	https://github.com/osgi/osgi-test
 * 	https://github.com/osgi/osgi-test/wiki
 * Examples: https://github.com/osgi/osgi-test/tree/main/examples
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
public class EPackageAIDiscoveryTest {

	private EObjectDiscoveryService<EPackage> discoveryService;
	private ManagementFactory factory;

	// Sample JSON data for testing
	private static final String SAMPLE_JSON_1 = """
			{
				"name": "TestSensor",
				"value": 42.5,
				"timestamp": "2025-01-01T12:00:00Z",
				"location": {
					"lat": 48.1234,
					"lon": 11.5678
				}
			}
			""";

	private static final String SAMPLE_JSON_2 = """
			{
				"name": "AnotherSensor",
				"value": 23.1,
				"timestamp": "2025-01-01T13:00:00Z",
				"location": {
					"lat": 48.9999,
					"lon": 11.1111
				}
			}
			""";

	private static final String SAMPLE_JSON_DIFFERENT = """
			{
				"deviceId": "sensor-123",
				"temperature": 25.5,
				"humidity": 60
			}
			""";

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@BeforeEach
	public void before(@InjectBundleContext BundleContext ctx, 
			@InjectService ServiceAware<EObjectDiscoveryService> discoveryAware) throws InterruptedException {
		discoveryService = discoveryAware.waitForService(2000l);
		assertNotNull(discoveryService);
		factory = ManagementFactory.eINSTANCE;
		
	}

	@AfterEach
	public void after() {
		
	}

	// ========== Fingerprint Generation Tests ==========

	@Test
	public void testCreateJsonFingerprint_ValidJson() throws Exception {
		Promise<String> promise = discoveryService.createJsonFingerprint(SAMPLE_JSON_1);
		String fingerprint = promise.getValue();

		assertNotNull(fingerprint, "Fingerprint should not be null for valid JSON");
		assertFalse(fingerprint.isEmpty(), "Fingerprint should not be empty");
	}

	@Test
	public void testCreateJsonFingerprint_SameStructureSameFingerprint() throws Exception {
		Promise<String> promise1 = discoveryService.createJsonFingerprint(SAMPLE_JSON_1);
		Promise<String> promise2 = discoveryService.createJsonFingerprint(SAMPLE_JSON_2);

		String fingerprint1 = promise1.getValue();
		String fingerprint2 = promise2.getValue();

		assertNotNull(fingerprint1);
		assertNotNull(fingerprint2);
		assertEquals(fingerprint1, fingerprint2,
				"JSON with same structure should produce same fingerprint");
	}

	@Test
	public void testCreateJsonFingerprint_DifferentStructureDifferentFingerprint() throws Exception {
		Promise<String> promise1 = discoveryService.createJsonFingerprint(SAMPLE_JSON_1);
		Promise<String> promise2 = discoveryService.createJsonFingerprint(SAMPLE_JSON_DIFFERENT);

		String fingerprint1 = promise1.getValue();
		String fingerprint2 = promise2.getValue();

		assertNotNull(fingerprint1);
		assertNotNull(fingerprint2);
		assertFalse(fingerprint1.equals(fingerprint2),
				"JSON with different structure should produce different fingerprints");
	}

	@Test
	public void testCreateJsonFingerprint_NullJson() throws Exception {

		assertInstanceOf(IllegalArgumentException.class, discoveryService.createJsonFingerprint(null).getFailure(), 
				"Passing a null json should result in an IllegalArgumentException");
	}

	@Test
	public void testCreateJsonFingerprint_EmptyJson() throws Exception {

		assertInstanceOf(IllegalArgumentException.class, discoveryService.createJsonFingerprint("").getFailure(), 
				"Passing an empty json should result in an IllegalArgumentException");
	}

	// ========== Generation Request Cache Tests ==========

	@Test
	public void testCacheGenerationRequest_Success() throws Exception {
		String fingerprint = "test-fingerprint-123";
		String requestId = "request-456";

		// Cache the request
		Promise<Void> cachePromise = discoveryService.cacheGenerationRequest(fingerprint, requestId);
		cachePromise.getValue(); // Wait for completion

		// Verify it's in progress
		Promise<Boolean> inProgressPromise = discoveryService.isGenerationInProgress(fingerprint);
		Boolean isInProgress = inProgressPromise.getValue();

		assertTrue(isInProgress, "Generation should be marked as in progress");
		
		//Cleanup
		discoveryService.removeGenerationRequestFromCache(fingerprint).getValue();
	}

	@Test
	public void testGetGenerationRequestIdForFingerprint_Success() throws Exception {
		String fingerprint = "test-fingerprint-789";
		String requestId = "request-101";

		// Cache the request
		discoveryService.cacheGenerationRequest(fingerprint, requestId).getValue();

		// Retrieve the request ID
		Promise<String> promise = discoveryService.getGenerationRequestIdForFingerprint(fingerprint);
		String retrievedRequestId = promise.getValue();

		assertEquals(requestId, retrievedRequestId,
				"Retrieved request ID should match cached request ID");
		
		//Cleanup
		discoveryService.removeGenerationRequestFromCache(fingerprint).getValue();
	}

	@Test
	public void testGetGenerationRequestIdForFingerprint_NotFound() throws Exception {
		String fingerprint = "non-existent-fingerprint";

		Promise<String> promise = discoveryService.getGenerationRequestIdForFingerprint(fingerprint);
		String requestId = promise.getValue();

		assertNull(requestId, "Request ID should be null for non-existent fingerprint");
	}

	@Test
	public void testIsGenerationInProgress_NotInProgress() throws Exception {
		String fingerprint = "non-existent-fingerprint";

		Promise<Boolean> promise = discoveryService.isGenerationInProgress(fingerprint);
		Boolean isInProgress = promise.getValue();

		assertFalse(isInProgress, "Generation should not be in progress for non-existent fingerprint");
	}

	@Test
	public void testRemoveGenerationRequestFromCache_Success() throws Exception {
		String fingerprint = "test-fingerprint-remove";
		String requestId = "request-remove";

		// Cache the request
		discoveryService.cacheGenerationRequest(fingerprint, requestId).getValue();

		// Verify it's in progress
		Boolean before = discoveryService.isGenerationInProgress(fingerprint).getValue();
		assertTrue(before, "Should be in progress before removal");

		// Remove from cache
		discoveryService.removeGenerationRequestFromCache(fingerprint).getValue();

		// Verify it's no longer in progress
		Boolean after = discoveryService.isGenerationInProgress(fingerprint).getValue();
		assertFalse(after, "Should not be in progress after removal");
	}

	// ========== Object Registration Cache Tests ==========

	@Test
	public void testCacheObjectRegistration_Success() throws Exception {

		String fingerprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_1).getValue();

		// Create test object metadata
		ObjectMetadata metadata = factory.createObjectMetadata();
		metadata.setObjectId("test-package-1");
		metadata.setObjectName("TestPackage");
		metadata.setObjectType("EPackage");
		metadata.setGenerationTriggerFingerprint(fingerprint);

		// Cache the registration
		Promise<Void> cachePromise = discoveryService.cacheObjectMetadata(metadata);
		cachePromise.getValue(); // Wait for completion

		// Verify it can be found by JSON pattern (using same fingerprint)
		Promise<ObjectMetadata> findPromise =
				discoveryService.findObjectByJsonPattern(SAMPLE_JSON_1, "test-channel", "EPackage");
		ObjectMetadata found = findPromise.getValue();

		assertNotNull(found);
		assertEquals("test-package-1", found.getObjectId(), "Retrieved ObjectRegistration should have the same objectId as the one we cached");
		
		//Cleanup
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();
	}

	@Test
	public void testRemoveObjectRegistrationFromCache_Success() throws Exception {

		String fingerprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_1).getValue();

		// Create test object metdata
		ObjectMetadata metadata = factory.createObjectMetadata();
		metadata.setObjectId("test-package-2");
		metadata.setObjectName("TestPackage2");
		metadata.setObjectType("EPackage");
		metadata.setGenerationTriggerFingerprint(fingerprint);

		// Cache the registration
		discoveryService.cacheObjectMetadata(metadata).getValue();

		// Remove from cache
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();

		// Try to find it (should return null since we removed it)
		Promise<ObjectMetadata> findPromise =
				discoveryService.findObjectByJsonPattern(SAMPLE_JSON_1, "test-channel", "EPackage");
		ObjectMetadata found = findPromise.getValue();
		assertNull(found, "We removed it from the cache so we should not find it anymore");
	}

	// ========== Find by JSON Pattern Tests ==========

	@Test
	public void testFindObjectByJsonPattern_NotFound() throws Exception {
		// Search for a pattern that doesn't exist
		Promise<ObjectMetadata> promise =
				discoveryService.findObjectByJsonPattern(SAMPLE_JSON_1, "test-channel", "EPackage");
		ObjectMetadata found = promise.getValue();

		assertNull(found, "Should return null when no matching package exists");
	}

	@Test
	public void testFindObjectByJsonPattern_NullJson() throws Exception {
		
		assertInstanceOf(IllegalArgumentException.class, discoveryService.findObjectByJsonPattern(null, "test-channel", "EPackage").getFailure(), 
				"Passing a null json should result in an IllegalArgumentException");
		
	}

	@Test
	public void testFindObjectByJsonPattern_EmptyJson() throws Exception {
		assertInstanceOf(IllegalArgumentException.class, discoveryService.findObjectByJsonPattern("", "test-channel", "EPackage").getFailure(), 
				"Passing an empty json should result in an IllegalArgumentException");

	}

	// ========== Search Similar Objects Tests ==========

	@Test
	public void testSearchSimilarObjects_ExactMatch() throws Exception {
		// Test exact match (similarity = 1.0)
		Promise<?> promise = discoveryService.searchSimilarObjects(SAMPLE_JSON_1, "EPackage", 1.0);
		Object result = promise.getValue();

		assertNotNull(result, "Should return result even if empty");
		assertTrue(result instanceof List, "Result should be a list");

		@SuppressWarnings("unchecked")
		List<ObjectMetadata> matches = (List<ObjectMetadata>) result;
		assertTrue(matches.isEmpty(), "Should be empty when no packages cached");
	}

	@Test
	public void testSearchSimilarObjects_FuzzyMatch() throws Exception {
		// Test fuzzy match (similarity < 1.0)
		Promise<?> promise = discoveryService.searchSimilarObjects(SAMPLE_JSON_1, "EPackage", 0.8);
		Object result = promise.getValue();

		assertNotNull(result, "Should return result even if empty");
		assertTrue(result instanceof List, "Result should be a list");

		@SuppressWarnings("unchecked")
		List<ObjectMetadata> matches = (List<ObjectMetadata>) result;
		assertTrue(matches.isEmpty(), "Should be empty when no packages cached");
	}

	@Test
	public void testSearchSimilarObjects_NullJson() throws Exception {
		
		assertInstanceOf(IllegalArgumentException.class, discoveryService.searchSimilarObjects(null, "EPackage", 0.9).getFailure(), 
				"Passing a null json should result in an IllegalArgumentException");
	}

	// ========== Integration Tests ==========

	@Test
	public void testFullWorkflow_CacheAndFind() throws Exception {
		// 1. Generate fingerprint
		Promise<String> fingerprintPromise = discoveryService.createJsonFingerprint(SAMPLE_JSON_1);
		String fingerprint = fingerprintPromise.getValue();
		assertNotNull(fingerprint);

		// 2. Cache a generation request
		String requestId = "workflow-request-123";
		discoveryService.cacheGenerationRequest(fingerprint, requestId).getValue();

		// 3. Verify it's in progress
		Boolean inProgress = discoveryService.isGenerationInProgress(fingerprint).getValue();
		assertTrue(inProgress);

		// 4. Get the request ID
		String retrievedRequestId = discoveryService.getGenerationRequestIdForFingerprint(fingerprint).getValue();
		assertEquals(requestId, retrievedRequestId);

		// 5. Create and cache an object registration
		ObjectMetadata metadata = factory.createObjectMetadata();
		metadata.setObjectId("workflow-package");
		metadata.setObjectName("WorkflowPackage");
		metadata.setObjectType("EPackage");
		metadata.setGenerationTriggerFingerprint(fingerprint);

		discoveryService.cacheObjectMetadata(metadata).getValue();

		// 6. Find by JSON pattern (should match by fingerprint)
		Promise<ObjectMetadata> findPromise =
				discoveryService.findObjectByJsonPattern(SAMPLE_JSON_2, "test-channel", "EPackage");
		ObjectMetadata found = findPromise.getValue();

		assertNotNull(found, "Should find the cached registration");
		assertEquals("workflow-package", found.getObjectId());
		assertEquals("WorkflowPackage", found.getObjectName());

		// 7. Clean up
		discoveryService.removeGenerationRequestFromCache(fingerprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();

		// 8. Verify cleanup
		Boolean afterCleanup = discoveryService.isGenerationInProgress(fingerprint).getValue();
		assertFalse(afterCleanup);
	}

	@Test
	public void testCacheGenerationRequest_NullValues() throws Exception {
		// Test with null fingerprint
		assertInstanceOf(IllegalArgumentException.class, discoveryService.cacheGenerationRequest(null, "request-id").getFailure(), 
				"Trying to cache a GenerationRequest with a null fingerprint should result in an IllegalArgumentException");
		
		// Test with null requestId
		assertInstanceOf(IllegalArgumentException.class, discoveryService.cacheGenerationRequest("fingerprint", null).getFailure(), 
				"Trying to cache a GenerationRequest with a null request id should result in an IllegalArgumentException");
	}

	@Test
	public void testCacheObjectRegistration_NullValues() throws Exception {
		// Test with null registration
		assertInstanceOf(IllegalArgumentException.class, discoveryService.cacheObjectMetadata(null).getFailure(),
				"Trying to cache a null ObjectMetadata should result in an IllegalArgumentException");
		

		// Test with registration without metadata
		ObjectMetadata metadata = factory.createObjectMetadata();
		assertInstanceOf(IllegalArgumentException.class, discoveryService.cacheObjectMetadata(metadata).getFailure(),
				"Trying to cache an ObjectMetadata without objectId should result in an IllegalArgumentException");
		
		// Test with registration with metadata but without fingerprint
		metadata.setObjectId("test-id");
		metadata.setObjectType("EPackage");
		assertInstanceOf(IllegalArgumentException.class, discoveryService.cacheObjectMetadata(metadata).getFailure(),
				"Trying to cache an ObjectMetadata without a triggered fingerprint in the ObjectMetadata should result in an IllegalArgumentException");
	}

}
