/**
 * Copyright (c) 2012 - 2023 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.management.ai.test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.fennec.ai.ecore.generator.component.EcoreAIGenerator;
import org.gecko.mac.mgmt.api.EObjectDiscoveryService;
import org.gecko.mac.mgmt.api.EObjectGenerationService;
import org.gecko.mac.mgmt.governanceapi.EObjectWorkflowService;
import org.gecko.mac.mgmt.management.GenerationRequest;
import org.gecko.mac.mgmt.management.GenerationStatus;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.management.ObjectStatus;
import org.gecko.osgi.messaging.MessagingService;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;
import org.osgi.util.promise.Promise;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;

/**
 * Comprehensive test suite for EPackageAIGenerationService.
 *
 * Tests cover:
 * - Generation request handling
 * - Duplicate request detection (fingerprint-based)
 * - Existing package registration detection
 * - AI generation success scenarios (with mocked EcoreAIGenerator)
 * - AI generation failure scenarios
 * - Generation status queries
 * - Active generation listing
 * - Cleanup job (CronJob)
 *
 * Note: EcoreAIGenerator is mocked to avoid using API credits.
 *
 * See documentation here:
 * 	https://github.com/osgi/osgi-test
 * 	https://github.com/osgi/osgi-test/wiki
 * Examples: https://github.com/osgi/osgi-test/tree/main/examples
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
public class EPackageAIGenerationTest {

	private EObjectGenerationService<EPackage> generationService;
	private EObjectDiscoveryService<EPackage> discoveryService;
	private ManagementFactory factory;
	private ServiceRegistration<EcoreAIGenerator> mockServiceRegistration;
	@SuppressWarnings("rawtypes")
	private ServiceRegistration<EObjectWorkflowService> mockWorkflowServiceRegistration;
	private ServiceRegistration<MessagingService> mockMsgServiceRegistration;

	@Mock
	private EcoreAIGenerator mockEcoreAIGenerator;
	
	@Mock
	private EObjectWorkflowService<EObject> workflowMock;
	
	@Mock
	private MessagingService mockedMsgService;
	
	// Sample JSON data for testing
	private static final String SAMPLE_JSON_SENSOR = """
			{
				"name": "TestSensor",
				"value": 42.5,
				"timestamp": "2025-01-01T12:00:00Z",
				"location": {
					"lat": 48.1234,
					"lon": 11.5678
				}
			}
			""";

	private static final String SAMPLE_JSON_DEVICE = """
			{
				"deviceId": "device-456",
				"status": "active",
				"metrics": {
					"temperature": 25.5,
					"humidity": 60
				}
			}
			""";
	private ProcessAuditSession session;

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@BeforeEach
	public void before(@InjectBundleContext BundleContext ctx,
			@InjectService(cardinality = 0) ServiceAware<EObjectGenerationService> generationAware,
			@InjectService(cardinality = 0) ServiceAware<EObjectDiscoveryService> discoveryAware,
			@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> auditManagerAware) throws Exception {

		// Register mock ChatCompletionService with highest priority
		Dictionary<String, Object> chatProps = new Hashtable<>();
		chatProps.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.MAX_VALUE);
		chatProps.put("component.name", "EcoreAIGenerator");

		mockServiceRegistration = ctx.registerService(
				EcoreAIGenerator.class,
				mockEcoreAIGenerator,
				chatProps);
		mockWorkflowServiceRegistration = ctx.registerService(
				EObjectWorkflowService.class,
				workflowMock,
				null);
		
		doNothing().when(mockedMsgService).publish(anyString(), any());

		// Register mock MessagingService with highest priority
		Dictionary<String, Object> msgProps = new Hashtable<>();
		msgProps.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.MAX_VALUE);
		msgProps.put("id", "biz");

		mockMsgServiceRegistration = ctx.registerService(
				MessagingService.class,
				mockedMsgService,
				msgProps);
		
		ProcessAuditSessionManager auditManager = auditManagerAware.waitForService(2000l);
		session = auditManager.startSession("osgi-test");

		generationService = generationAware.waitForService(2000L);
		assertNotNull(generationService, "Generation service should be available");

		discoveryService = discoveryAware.waitForService(2000L);
		assertNotNull(discoveryService, "Discovery service should be available");

		factory = ManagementFactory.eINSTANCE;

	}

	@AfterEach
	public void after() {
		session.complete("osgi-test");
		if (mockWorkflowServiceRegistration != null) {
			mockWorkflowServiceRegistration.unregister();
		}
		if(mockServiceRegistration != null) {
			mockServiceRegistration.unregister();
		}
		if(mockMsgServiceRegistration != null) {
			mockMsgServiceRegistration.unregister();
		}
	}



	// ========== Helper Methods ==========

	/**
	 * Create a mock EPackage for testing.
	 */
	private EPackage createMockEPackage(String name, String nsURI, String nsPrefix) {
		EPackage ePackage = EcoreFactory.eINSTANCE.createEPackage();
		ePackage.setName(name);
		ePackage.setNsURI(nsURI);
		ePackage.setNsPrefix(nsPrefix);

		// Add a simple EClass to make it realistic
		EClass eClass = EcoreFactory.eINSTANCE.createEClass();
		eClass.setName("DataClass");
		EAttribute attr = EcoreFactory.eINSTANCE.createEAttribute();
		attr.setName("value");
		attr.setEType(EcorePackage.Literals.ESTRING);
		eClass.getEStructuralFeatures().add(attr);
		ePackage.getEClassifiers().add(eClass);

		return ePackage;
	}

	// ========== Request Generation Tests ==========

	@Test
	public void testRequestGeneration_Success() throws Exception {
		assertNotNull(generationService, "Generation service should be available");

		assertNotNull(discoveryService, "Discovery service should be available");
		// Setup mock to return a generated EPackage
		EPackage mockPackage = createMockEPackage("TestSensorPackage",
				"http://test.sensor/1.0", "testsensor");
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage);
		
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");

		// Request generation
		Promise<String> requestPromise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "test-channel", "test-user", "EPackage", parentPackage);
		String requestId = requestPromise.getValue();

		assertNotNull(requestId, "Request ID should not be null");
		assertFalse(requestId.isEmpty(), "Request ID should not be empty");

		// Wait a bit for async generation to complete
		Thread.sleep(1000);

		// Check generation status
		Promise<GenerationRequest> statusPromise = generationService.getGenerationStatus(requestId);
		GenerationRequest status = statusPromise.getValue();

		assertNotNull(status, "Generation request should be found");
		assertEquals(requestId, status.getRequestId());
		assertEquals("test-channel", status.getSourceChannel());
		assertEquals("test-user", status.getRequestingUser());
		assertEquals(SAMPLE_JSON_SENSOR, status.getJsonSample());

		// Status should eventually be COMPLETED
		assertTrue(status.getStatus() == GenerationStatus.COMPLETED ||
				status.getStatus() == GenerationStatus.IN_PROGRESS,
				"Status should be COMPLETED or IN_PROGRESS");
		
//		Cleanup
		String fingertprint = (String) discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();
		
	}

	@Test
	public void testRequestGeneration_DuplicateRequest() throws Exception {
		// Setup mock to return a generated EPackage
		EPackage mockPackage = createMockEPackage("SensorPackage",
				"http://sensor/1.0", "sensor");
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage);
		
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");


		// First request
		Promise<String> request1Promise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "channel1", "user1", "EPackage", parentPackage);
		String requestId1 = request1Promise.getValue();
		assertNotNull(requestId1);

		// Second request with same JSON structure (should detect in-progress)
		// Note: We need to request before the first one completes
		Promise<String> request2Promise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "channel2", "user2", "EPackage", parentPackage);
		String requestId2 = request2Promise.getValue();

		// Both should return the same request ID since generation is in progress
		assertEquals(requestId1, requestId2,
				"Duplicate requests should return the same request ID");

		// Clean up - wait for generation to complete
		Thread.sleep(1000);
		
//		Cleanup
		String fingertprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();
	}

	@Test
	public void testRequestGeneration_ExistingObjectRegistration() throws Exception {
		// First, create a fingerprint and cache an ObjectRegistration
		String fingerprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_DEVICE).getValue();

		ObjectMetadata metadata = factory.createObjectMetadata();
		metadata.setObjectId("existing-package-id");
		metadata.setObjectName("ExistingPackage");
		metadata.setObjectType("EPackage");
		metadata.setStatus(ObjectStatus.DRAFT);
		metadata.setGenerationTriggerFingerprint(fingerprint);

		discoveryService.cacheObjectMetadata(metadata).getValue();

		// Now request generation with the same JSON structure
		Promise<String> requestPromise = generationService.requestGeneration(
				SAMPLE_JSON_DEVICE, "test-channel", "test-user", "EPackage", null);
		String result = requestPromise.getValue();

		// Should return the existing package ID, not trigger new generation
		assertEquals("existing-package-id", result,
				"Should return existing package registration ID");

		// Clean up
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();
	}

	@Test
	public void testRequestGeneration_AIGenerationFailure() throws Exception {
		// Setup mock to throw exception (simulating AI failure)
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenThrow(new RuntimeException("AI generation failed"));
		
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");

		// Request generation
		Promise<String> requestPromise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "test-channel", "test-user", "EPackage", parentPackage);
		String requestId = requestPromise.getValue();

		assertNotNull(requestId);

		// Wait for async generation to fail
		Thread.sleep(1000);

		// Check generation status
		Promise<GenerationRequest> statusPromise = generationService.getGenerationStatus(requestId);
		GenerationRequest status = statusPromise.getValue();

		assertNotNull(status);
		assertTrue(status.getStatus() == GenerationStatus.FAILED ||
				status.getStatus() == GenerationStatus.IN_PROGRESS,
				"Status should be FAILED or still IN_PROGRESS");

		if (status.getStatus() == GenerationStatus.FAILED) {
			assertNotNull(status.getErrorMessage(), "Error message should be present on failure");
		}
//		Cleanup
		String fingertprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();
	}



	// ========== Get Generation Status Tests ==========

	@Test
	public void testGetGenerationStatus_ExistingRequest() throws Exception {
		// Setup mock
		EPackage mockPackage = createMockEPackage("StatusTestPackage",
				"http://statustest/1.0", "statustest");
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage);
		
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");

		// Create a generation request
		Promise<String> requestPromise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "test-channel", "test-user", "EPackage", parentPackage);
		String requestId = requestPromise.getValue();

		// Query status
		Promise<GenerationRequest> statusPromise = generationService.getGenerationStatus(requestId);
		GenerationRequest status = statusPromise.getValue();

		assertNotNull(status, "Status should not be null");
		assertEquals(requestId, status.getRequestId());
		assertNotNull(status.getRequestTime());
		assertNotNull(status.getStatus());

		// Wait for completion
		Thread.sleep(1000);
		
//		Cleanup
		String fingertprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();
		
	}

	@Test
	public void testGetGenerationStatus_NonExistentRequest() throws Exception {
		String fakeRequestId = "non-existent-request-123";

		Promise<GenerationRequest> statusPromise = generationService.getGenerationStatus(fakeRequestId);
		GenerationRequest status = statusPromise.getValue();

		assertNull(status, "Status should be null for non-existent request");
	}

	// ========== List Active Generations Tests ==========

	@Test
	public void testListActiveGenerations_MultipleRequests() throws Exception {
		EPackage parentPackage1 = createMockEPackage("ParentPackage1",
				"http://parent1/1.0", "parent1");

		EPackage parentPackage2 = createMockEPackage("ParentPackage2",
				"http://parent2/1.0", "parent2");

		// Setup mock to return packages
		EPackage mockPackage1 = createMockEPackage("Package1", "http://pkg1/1.0", "pkg1");
		EPackage mockPackage2 = createMockEPackage("Package2", "http://pkg2/1.0", "pkg2");

		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage1)
		.thenReturn(mockPackage2);

		// Create multiple generation requests with different JSON

		Promise<String> request1 = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "channel1", "user1", "EPackage", parentPackage1);
		Promise<String> request2 = generationService.requestGeneration(
				SAMPLE_JSON_DEVICE, "channel2", "user2", "EPackage", parentPackage2);

		String requestId1 = request1.getValue();
		String requestId2 = request2.getValue();

		assertNotNull(requestId1);
		assertNotNull(requestId2);

		// List active generations (might include both or just one depending on timing)
		Promise<List<GenerationRequest>> listPromise = generationService.listActiveGenerations();
		List<GenerationRequest> activeGenerations = listPromise.getValue();

		assertNotNull(activeGenerations);
		// At least one should be active (or both might have completed already)
		assertTrue(activeGenerations.size() >= 0,
				"Active generations list should be non-negative");

		// Wait for completions
		Thread.sleep(2000);

		// After completion, active list should be smaller or empty
		Promise<List<GenerationRequest>> listPromise2 = generationService.listActiveGenerations();
		List<GenerationRequest> activeGenerations2 = listPromise2.getValue();
		assertTrue(activeGenerations2.size() <= activeGenerations.size(),
				"Active generations should decrease after completion");
		
//		Cleanup
		String fingertprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();
		
		fingertprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_DEVICE).getValue();
		discoveryService.removeGenerationRequestFromCache(fingertprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingertprint).getValue();		
	}

	@Test
	public void testListActiveGenerations_EmptyList() throws Exception {
		// No active generations initially
		Promise<List<GenerationRequest>> listPromise = generationService.listActiveGenerations();
		List<GenerationRequest> activeGenerations = listPromise.getValue();

		assertNotNull(activeGenerations, "List should not be null");
		// List might be empty or contain some in-progress items from previous tests
	}


	// ========== Workflow Service Integration Tests ==========

	@Test
	public void testWorkflowServiceUploadDraftCalled() throws Exception {
		// Setup mock to return a generated EPackage
		EPackage mockPackage = createMockEPackage("TestWorkflowPackage",
				"http://test.workflow/1.0", "testworkflow");
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage);
		
		// Mock the workflow service to return a successful Promise
		when(workflowMock.uploadDraft(any(EPackage.class), any(ObjectMetadata.class)))
		.thenReturn(null); // The actual service returns a Promise<String>, but we don't use the return value
		
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");

		// Request generation
		Promise<String> requestPromise = generationService.requestGeneration(
				SAMPLE_JSON_SENSOR, "test-channel", "test-user", "EPackage", parentPackage);
		String requestId = requestPromise.getValue();

		assertNotNull(requestId, "Request ID should not be null");

		// Wait for async generation to complete
		Thread.sleep(2000);

		// Verify that uploadDraft was called on the workflow service
		// We verify it was called with:
		// 1. The generated EPackage (mockPackage)
		// 2. An ObjectMetadata with the correct properties
		verify(workflowMock).uploadDraft(
				eq(mockPackage), 
				any(ObjectMetadata.class)
		);

		// Additional verification: check the metadata properties passed to uploadDraft
		verify(workflowMock).uploadDraft(
				any(EPackage.class),
				org.mockito.ArgumentMatchers.argThat(metadata -> 
					metadata.getObjectId().equals(requestId) &&
					metadata.getObjectName().equals("TestWorkflowPackage") &&
					metadata.getSourceChannel().equals("AI_GENERATOR") &&
					metadata.getObjectType().equals("EPackage") &&
					metadata.getVersion().equals("1.0.0") &&
					metadata.getObjectRef() == mockPackage
				)
		);

		// Cleanup
		String fingerprint = discoveryService.createJsonFingerprint(SAMPLE_JSON_SENSOR).getValue();
		discoveryService.removeGenerationRequestFromCache(fingerprint).getValue();
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();
	}

	// ========== Integration Tests ==========

	@Test
	public void testFullGenerationWorkflow() throws Exception {
		EPackage parentPackage = createMockEPackage("ParentPackage",
				"http://parent/1.0", "parent");
		// Setup mock
		EPackage mockPackage = createMockEPackage("WorkflowPackage",
				"http://workflow/1.0", "workflow");
		when(mockEcoreAIGenerator.generateEcoreForUnknownJson(anyString(), any(EPackage.class)))
		.thenReturn(mockPackage);

		String testJson = """
				{"workflow": "test", "step": 1}
				""";

		// 1. Request generation
		Promise<String> requestPromise = generationService.requestGeneration(
				testJson, "workflow-channel", "workflow-user", "EPackage", parentPackage);
		String requestId = requestPromise.getValue();
		assertNotNull(requestId);

		// 2. Check initial status (should be REQUESTED or IN_PROGRESS)
		GenerationRequest initialStatus = generationService.getGenerationStatus(requestId).getValue();
		assertNotNull(initialStatus);
		assertTrue(initialStatus.getStatus() == GenerationStatus.REQUESTED ||
				initialStatus.getStatus() == GenerationStatus.IN_PROGRESS);

		// 3. Wait for completion
		Thread.sleep(2000);

		// 4. Check final status (should be COMPLETED)
		GenerationRequest finalStatus = generationService.getGenerationStatus(requestId).getValue();
		assertNotNull(finalStatus);
		assertEquals(GenerationStatus.COMPLETED, finalStatus.getStatus(),
				"Final status should be COMPLETED");
		assertNotNull(finalStatus.getCompletionTime());
		assertNotNull(finalStatus.getResultPackageId());

		// 5. Verify ObjectRegistration was cached in discovery service
		String fingerprint = discoveryService.createJsonFingerprint(testJson).getValue();
		ObjectMetadata metadata = discoveryService
				.findObjectByJsonPattern(testJson, "workflow-channel", "EPackage")
				.getValue();

		assertNotNull(metadata, "ObjectRegistration should be cached");
		assertEquals("WorkflowPackage", metadata.getObjectName());
		assertEquals(ObjectStatus.DRAFT, metadata.getStatus());

		// 6. Clean up
		discoveryService.removeObjectRegistrationFromCache(fingerprint).getValue();
	}
}
