/*
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Data In Motion - initial API and implementation
 */
package org.gecko.mac.management.ai;

import static java.util.Objects.requireNonNull;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.ecore.EPackage;
import org.eclipse.fennec.ecore.fingerprint.generator.EcoreFingerprintGenerator;
import org.gecko.mac.mgmt.api.EObjectDiscoveryService;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

/**
 * Implementation of the EPackageDiscoveryService for JSON pattern recognition
 * and package discovery.
 *
 * This service is a singleton and it is responsible to cache the ObjectMetadata and their statuses
 * until the EPackage is actually into the system and can be used
 * 
 * This service also provides:
 * - JSON fingerprinting to identify structurally identical data
 * - Package lookup by JSON structure
 * - Generation request caching to prevent duplicate AI calls
 * - Similarity search for finding related packages
 * 

 */
@Component(immediate = true, name = "EPackageAIDiscoveryService", scope = ServiceScope.SINGLETON)
public class EPackageAIDiscoveryService implements EObjectDiscoveryService<EPackage> {

	private static final Logger LOGGER = Logger.getLogger(EPackageAIDiscoveryService.class.getName());

	/**
	 * Cache of generation requests indexed by fingerprint
	 * Key: JSON fingerprint hash
	 * Value: GenerationRequest model object
	 */
	private final Map<String, String> generationCache = new ConcurrentHashMap<>();
	
	/** objMetadataCache 
	 *  Cache ObjectMetadata indexed by JSON fingerprint hash of the json which triggered the generation of this EPackage
	 *  Key: JSON fingerprint hash of the json which triggered the generation of this EPackage
	 *  Value: ObjectMetadata
	 * 
	 * */
	private final Map<String, ObjectMetadata> objMetadataCache = new ConcurrentHashMap<>();

	/**
	 * Promise factory for creating async operations
	 */
	private final PromiseFactory promiseFactory;

	@Activate
	public EPackageAIDiscoveryService() {
		// Create promise factory with default executor
		this.promiseFactory = new PromiseFactory(Executors.newCachedThreadPool());
		LOGGER.info("EPackageDiscoveryService activated");
	}

	@Deactivate
	public void deactivate() {
		generationCache.clear();
		objMetadataCache.clear();
		LOGGER.info("EPackageDiscoveryService deactivated");
	}

	/**
	 * Find existing EPackage that matches JSON structure.
	 *
	 * Algorithm:
	 * 1. Generate fingerprint from JSON sample
	 * 2. Query storage service for packages with matching fingerprint
	 * 3. Filter by sourceChannel if provided
	 * 4. Return first matching package or null
	 */
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#findObjectByJsonPattern(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Promise<ObjectMetadata> findObjectByJsonPattern(String jsonSample, String sourceChannel, String targetType) {
		
		return promiseFactory.submit(() -> {
			if (jsonSample == null || jsonSample.trim().isEmpty()) {
				LOGGER.warning("Cannot find package for null or empty JSON sample");
				throw new IllegalArgumentException("Cannot find package for null or empty JSON sample");
			}

			try {
				// Generate fingerprint from JSON (without parent EClass for discovery)
				String fingerprint = EcoreFingerprintGenerator.generateFingerprint(jsonSample, null);

				if (fingerprint == null) {
					LOGGER.severe("Failed to generate fingerprint from JSON sample");
					return null;
				}

				LOGGER.info(String.format("Generated fingerprint %s for JSON pattern lookup", fingerprint));

				// Find packages with matching fingerprint in metadata
				List<ObjectMetadata> matchingPackages = findPackagesByFingerprint(fingerprint);

				if (!matchingPackages.isEmpty()) {
					ObjectMetadata match = matchingPackages.get(0);
					LOGGER.info(String.format("Found matching package: (id: %s)", match.getObjectId()));
					return match;
				}

				LOGGER.info(String.format("No matching package found for fingerprint %s", fingerprint));
				return null;

			} catch (Exception e) {
				LOGGER.log(Level.SEVERE, "Error during package lookup by JSON pattern", e);
				throw e;
			}
		});
	}
	
	/**
	 * Search for packages with similar JSON patterns.
	 *
	 * Currently implements exact fingerprint matching.
	 * Future enhancement: implement similarity scoring based on:
	 * - Common fields
	 * - Type compatibility
	 * - Structural similarity
	 */
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#searchSimilarObjects(java.lang.String, java.lang.String, double)
	 */
	/**
	 * @param jsonSample
	 * @param targetType
	 * @param similarityThreshold
	 * @return
	 */
	@Override
	public Promise<List<ObjectMetadata>> searchSimilarObjects(String jsonSample, String targetType, double similarityThreshold) {
		return promiseFactory.submit(() -> {
			if (jsonSample == null || jsonSample.trim().isEmpty()) {
				LOGGER.warning("Cannot search for similar packages with null or empty JSON sample");
				throw new IllegalArgumentException("Cannot search for similar packages with null or empty JSON sample");
			}
			
			try {
				// Generate fingerprint from JSON
				String fingerprint = EcoreFingerprintGenerator.generateFingerprint(jsonSample, null);
				
				if (fingerprint == null) {
					LOGGER.severe("Failed to generate fingerprint from JSON sample");
					return Collections.emptyList();
				}
				
				// For now, implement exact match (similarity = 1.0)
				// Future: implement fuzzy matching with similarity scoring
				if (similarityThreshold >= 1.0) {
					List<ObjectMetadata>  exactMatches = findPackagesByFingerprint(fingerprint);
					LOGGER.info(String.format("Found %d packages with exact match (similarity=1.0)",
							exactMatches.size()));
					return exactMatches;
				}
				
				// Fuzzy matching for similarity < 1.0	
				return findPackagesBySimilarFingerprint(fingerprint, similarityThreshold);
				
			} catch (Exception e) {
				LOGGER.log(Level.SEVERE, "Error during similar package search", e);
				throw e;
			}
		});
	}


	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#createJsonFingerprint(java.lang.String)
	 */
	@Override
	public Promise<String> createJsonFingerprint(String jsonSample) {
		return promiseFactory.submit(() -> {
			if (jsonSample == null || jsonSample.trim().isEmpty()) {
				LOGGER.severe("Cannot create fingerprint from null or empty JSON sample");
				throw new IllegalArgumentException(String.format("Cannot create fingerprint from null or empty JSON sample"));
			}
			try {
				// Use EcoreFingerprintGenerator to create structural hash
				String fingerprint = EcoreFingerprintGenerator.generateFingerprint(jsonSample, null);
				return fingerprint;
			} catch (Exception e) {
				LOGGER.log(Level.SEVERE, "Error during fingerprint creation", e);
				throw e;
			}
		});
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#isGenerationInProgress(java.lang.String)
	 */
	@Override
	public Promise<Boolean> isGenerationInProgress(String jsonFingerprint) {
		return promiseFactory.submit(() -> {
			if (jsonFingerprint == null || jsonFingerprint.trim().isEmpty()) {
				return false;
			}

			String requestId = generationCache.getOrDefault(jsonFingerprint, null);
			if(requestId == null) return false;
			return true;
			
		});
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#cacheGenerationRequest(java.lang.String, java.lang.String)
	 */
	@Override
	public Promise<Void> cacheGenerationRequest(String jsonFingerprint, String requestId) {
		
		return promiseFactory.submit(() -> {
			if (jsonFingerprint == null || requestId == null) {
				LOGGER.severe("Cannot cache generation request with null fingerprint or requestId");
				throw new IllegalArgumentException("Cannot cache generation request with null fingerprint or requestId");
			}

			generationCache.put(jsonFingerprint, requestId);

			LOGGER.info(String.format("Cached generation request: fingerprint=%s, requestId=%s",
					jsonFingerprint, requestId));

			return null;
		});
		
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#getGenerationRequestIdForFingerprint(java.lang.String)
	 */
	@Override
	public Promise<String> getGenerationRequestIdForFingerprint(String jsonFingerprint) {
		return promiseFactory.submit(() -> {
			return generationCache.getOrDefault(jsonFingerprint, null);
		});
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#removeGenerationRequestFromCache(java.lang.String)
	 */
	@Override
	public Promise<Void> removeGenerationRequestFromCache(String fingerprint) {
		return promiseFactory.submit(() -> {
			
			generationCache.remove(fingerprint);
			
			LOGGER.info(String.format("Removed generation request from cache: fingerprint=%s",
					fingerprint));
			
			return null;
		});
	
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#cacheObjectRegistration(org.gecko.mac.mgmt.management.ObjectRegistration)
	 */
	@Override
	public Promise<Void> cacheObjectMetadata(ObjectMetadata objectMetadata) {
		
		return promiseFactory.submit(() -> {
			if (objectMetadata == null || objectMetadata.getObjectId() == null || objectMetadata.getGenerationTriggerFingerprint() == null) {
				LOGGER.warning("Cannot cache null ObjectMetadata or with null metadata or with null generation trigger fingerprint");
				throw new IllegalArgumentException("Cannot cache null ObjectMetadata or with null metadata or with null generation trigger fingerprint");
			}

			objMetadataCache.put(objectMetadata.getGenerationTriggerFingerprint(), objectMetadata);

			LOGGER.info(String.format("Cached ObjectMetadata: fingerprint=%s",
					objectMetadata.getGenerationTriggerFingerprint()));

			return null;
		});
	}


	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectDiscoveryService#removeObjectRegistrationFromCache(java.lang.String)
	 */
	@Override
	public Promise<Void> removeObjectRegistrationFromCache(String fingerprint) {
		return promiseFactory.submit(() -> {
			
			objMetadataCache.remove(fingerprint);
			
			LOGGER.info(String.format("Removed ObjectMetadata from cache: fingerprint=%s",
					fingerprint));
			
			return null;
		});
	}
	
	// ========== Helper Methods ==========

		/**
		 * Find EPackage by fingerprint in their metadata. There should be one actually.
		 */
		private List<ObjectMetadata> findPackagesByFingerprint(String fingerprint) {	
			requireNonNull(fingerprint, "Fingerprint must not be null");
			if(objMetadataCache.containsKey(fingerprint)) {
				return List.of(objMetadataCache.get(fingerprint));
			}
			return Collections.emptyList();
		}
		
		/**
		 * Find EPackages that have a similar triggered fingerprint.
		 */
		private List<ObjectMetadata> findPackagesBySimilarFingerprint(String fingerprint, double similarityThreshold) {	
			requireNonNull(fingerprint, "Fingerprint must not be null");
			return objMetadataCache.entrySet().stream()
					.filter(e -> EcoreFingerprintGenerator.computeHashDistance(fingerprint, e.getKey()) >= similarityThreshold)
					.map(e -> e.getValue()).toList();			
		}

}
