/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.management.ai;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.fennec.ai.ecore.generator.component.EcoreAIGenerator;
import org.gecko.mac.audit.ActionType;
import org.gecko.mac.audit.Audit;
import org.gecko.mac.audit.CategoryType;
import org.gecko.mac.audit.helper.AuditHelper;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.mac.mgmt.api.EObjectDiscoveryService;
import org.gecko.mac.mgmt.api.EObjectGenerationService;
import org.gecko.mac.mgmt.governanceapi.EObjectWorkflowService;
import org.gecko.mac.mgmt.management.GenerationRequest;
import org.gecko.mac.mgmt.management.GenerationStatus;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.typedevent.TypedEventBus;
import org.osgi.service.typedevent.annotations.RequireTypedEvent;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

/**
 * 
 * @author ilenia
 * @since Oct 8, 2025
 */
@RequireTypedEvent
@Component(name = "EPackageAIGenerationService", service = {EObjectGenerationService.class}, 
configurationPid = "EPackageAIGenerationService", configurationPolicy = ConfigurationPolicy.OPTIONAL)
public class EPackageAIGenerationService implements EObjectGenerationService<EPackage>, Runnable {
	
	@Reference
	private ProcessAuditSessionManager sessionManager;

	@Reference
	private EcoreAIGenerator ecoreAIGenerator;

	@Reference(target = "(component.name=EPackageAIDiscoveryService)")
	private EObjectDiscoveryService<EPackage> packageDiscoveryService;
	
	@Reference
	private EObjectWorkflowService<EObject> workflowService;
	
	@Reference
	TypedEventBus typedEventBus;

	private static final Logger LOGGER = Logger.getLogger(EPackageAIGenerationService.class.getName());

	/**
	 * Promise factory for creating async operations
	 */
	private final PromiseFactory promiseFactory;

	private ConcurrentHashMap<String, GenerationRequest> cachedGenerationRequests = new ConcurrentHashMap<>();

	private ScheduledExecutorService executorService;

	@Activate
	public EPackageAIGenerationService() {
		
		// Create promise factory with default executor
		this.promiseFactory = new PromiseFactory(Executors.newCachedThreadPool());
		LOGGER.info("EPackageGenerationService activated");
		executorService = Executors.newScheduledThreadPool(1);
		executorService.scheduleAtFixedRate(this::run, 0, 60, TimeUnit.MINUTES);
	}
	
	@Deactivate
	public void deactivate() {
		executorService.shutdown();
	}


	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectGenerationService#requestGeneration(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Promise<String> requestGeneration(String jsonSample, String sourceChannel, String requestingUser, String targetType) {

		return requestGeneration(jsonSample, sourceChannel, requestingUser, targetType, null);
	}



	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EObjectGenerationService#requestGeneration(java.lang.String, java.lang.String, java.lang.String, java.lang.String, org.eclipse.emf.ecore.EObject)
	 */
	@Override
	public Promise<String> requestGeneration(String jsonSample, String sourceChannel, String requestingUser, String targetType, EPackage parentEPackage) {

		ProcessAuditSession currentSession = retrieveCurrentSession();
		
		return promiseFactory.submit(() -> {
			
			sessionManager.setCurrentSession(currentSession);

			//			Here we have to create the json fingerprint 
			String fingerprint = packageDiscoveryService.createJsonFingerprint(jsonSample).getValue();

			// Synchronize the entire duplicate detection and cache operation to prevent race conditions
			synchronized (this) {
				//			If a generation is already in progress we return the GenerationRequest#id so one can query for the generation status
				if(packageDiscoveryService.isGenerationInProgress(fingerprint).getValue()) {
					LOGGER.info(String.format("Generation is already in progress for this json fingerprint."));
					String existingRequestId = packageDiscoveryService.getGenerationRequestIdForFingerprint(fingerprint).getValue();
					if (existingRequestId != null) {
						return existingRequestId;
					} else {
						LOGGER.warning("Generation was marked as in progress but no request ID found for fingerprint: " + fingerprint);
					}
				}

				//			If no generation is in progress than we have to check whether an ObjectRegistration has already been created for that
				//			(this may be the case when an EPackage draft is available but not yet approved)
				ObjectMetadata metadata = packageDiscoveryService.findObjectByJsonPattern(jsonSample, sourceChannel, targetType).getValue();
				if(metadata != null) {
					LOGGER.info(String.format("An ObjectMetadata for this jsonSample already exists with id %s", metadata.getObjectId()));
					return metadata.getObjectId();
				}

				//			If we end up here it means we really have to generate a new EPackage

				//			1. We then create a GenerationRequest to store in the DiscoveryService for as long as the generation takes
				GenerationRequest generationRequest = ManagementFactory.eINSTANCE.createGenerationRequest();
				generationRequest.setRequestId(UUID.randomUUID().toString());
				generationRequest.setSourceChannel(sourceChannel);
				generationRequest.setRequestingUser(requestingUser);
				generationRequest.setRequestTime(Instant.now());
				generationRequest.setJsonSample(jsonSample);
				generationRequest.setStatus(GenerationStatus.REQUESTED);			
				
				// Cache the generation request synchronously within the synchronized block
				try {
					packageDiscoveryService.cacheGenerationRequest(fingerprint, generationRequest.getRequestId()).getValue();
				} catch (Exception e) {
					LOGGER.log(Level.SEVERE, "Failed to cache generation request", e);
					throw new RuntimeException("Failed to cache generation request", e);
				}

				//		    3. We save the GenerationRequest also in the GenerationService cache so one can query for the status
				cachedGenerationRequests.put(generationRequest.getRequestId(), generationRequest);
				
				// Start the async generation outside the synchronized block
				startAsyncGeneration(generationRequest, fingerprint, jsonSample, parentEPackage);
				
				return generationRequest.getRequestId();
			}
		});
	}
	
	/**
	 * Start async AI generation outside the synchronized block to avoid blocking other requests
	 */
	private void startAsyncGeneration(GenerationRequest generationRequest, String fingerprint, String jsonSample, EPackage parentEPackage) {
		ProcessAuditSession currentSession = retrieveCurrentSession();
		
		//			2. We trigger the actual AI generation in another thread so we do not block here
		promiseFactory.submit(() -> {
			sessionManager.setCurrentSession(currentSession);
			generationRequest.setStartTime(Instant.now());
			generationRequest.setStatus(GenerationStatus.IN_PROGRESS);
			return ecoreAIGenerator.generateEcoreForUnknownJson(jsonSample, parentEPackage);
		}).onSuccess(ep -> { 
			sessionManager.setCurrentSession(currentSession);
			//			2.1 We create the ObjectRegistration if the AI generation was successfull and we store that in the DiscoveryService
			ObjectMetadata objMetadata = ManagementFactory.eINSTANCE.createObjectMetadata();
			objMetadata.setObjectId(generationRequest.getRequestId());
			objMetadata.setObjectName(ep.getName());
			objMetadata.setSourceChannel("AI_GENERATOR");
			objMetadata.setObjectType("EPackage");
			objMetadata.setGenerationTriggerFingerprint(fingerprint);
//			objMetadata.setStatus(ObjectStatus.DRAFT);
			objMetadata.setVersion("1.0.0");
			objMetadata.setObjectRef(ep);
			objMetadata.getProperties().put("file.extension", "ecore");
			objMetadata.setUploadUser("AI_GENERATOR");
			workflowService.uploadDraft(ep, objMetadata);
			packageDiscoveryService.cacheObjectMetadata(objMetadata);				

			//			2.2 And we update the status and timing of the GenerationRequest and we are removing it from the DiscoveryService cache
			generationRequest.setStatus(GenerationStatus.COMPLETED);
			generationRequest.setCompletionTime(Instant.now());
			generationRequest.setResultPackageId(ep.getNsURI()); 
			packageDiscoveryService.removeGenerationRequestFromCache(fingerprint);
			Audit audit = AuditHelper.createAudit(ep.getNsURI(), Instant.now().toString(), "eclipse.fennec.ai.generator",
					ActionType.GENERATED, CategoryType.MODEL, "EPackage has been generated via AI");
			typedEventBus.deliver("audit", audit);
			sessionManager.getCurrentSession().checkpoint("EPackage has been generated via AI", audit);
			sessionManager.getCurrentSession().complete("ai-generation completed");
		}).onFailure(t -> {		
			sessionManager.setCurrentSession(currentSession);
			//			2.3 On failure we only have to update the GenerationRequest and we are removing it from the DiscoveryService cache
			generationRequest.setStatus(GenerationStatus.FAILED);
			generationRequest.setCompletionTime(Instant.now());
			generationRequest.setErrorMessage(t.getMessage());
			packageDiscoveryService.removeGenerationRequestFromCache(fingerprint);
			Audit audit = AuditHelper.createAudit(generationRequest.getRequestId(), Instant.now().toString(), "eclipse.fennec.ai.generator",
					ActionType.FAILURE, CategoryType.MODEL, "Failed to generate EPackage via AI");
			typedEventBus.deliver("audit", audit);
			sessionManager.getCurrentSession().checkpoint("Failed to generate EPackage via AI", audit);
			sessionManager.getCurrentSession().complete("ai-generation completed");
		});
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EPackageGenerationService#getGenerationStatus(java.lang.String)
	 */
	@Override
	public Promise<GenerationRequest> getGenerationStatus(String requestId) {

		return promiseFactory.submit(() -> {
			return cachedGenerationRequests.getOrDefault(requestId, null);
		});
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EPackageGenerationService#cancelGeneration(java.lang.String)
	 */
	@Override
	public Promise<Boolean> cancelGeneration(String requestId) {
		// TODO Auto-generated method stub
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.mac.mgmt.api.EPackageGenerationService#listActiveGenerations()
	 */
	@Override
	public Promise<List<GenerationRequest>> listActiveGenerations() {

		return promiseFactory.submit(() -> {
			return cachedGenerationRequests.values().stream().filter(gr -> GenerationStatus.IN_PROGRESS.equals(gr.getStatus())).toList();
		});
	}



	/* 
	 * (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run()  {
		cachedGenerationRequests.entrySet().removeIf(e -> 
		!GenerationStatus.IN_PROGRESS.equals(e.getValue().getStatus()) && e.getValue().getCompletionTime().isBefore(Instant.now().minus(10, ChronoUnit.MINUTES)));
	}
	
	private ProcessAuditSession retrieveCurrentSession() {
		ProcessAuditSession currentSession = sessionManager.getCurrentSession();
		if(currentSession != null) {
			return currentSession;
		} else {
			LOGGER.severe(String.format("SESSION IS NULL FOR AUDITING!"));
			throw new IllegalStateException(String.format("SESSION IS NULL FOR AUDITING!"));
		}
	}

}
