/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.management.minio.emf;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

import io.minio.MinioClient;
import io.minio.PutObjectArgs;

/**
 * OutputStream implementation for MinIO that buffers data in memory
 * and uploads on close().
 * 
 * For large files, this could be optimized to use streaming uploads
 * or temporary files, but for typical EMF models this approach is sufficient.
 */
class MinioOutputStream extends ByteArrayOutputStream {
    
    private final MinioClient minioClient;
    private final String bucket;
    private final String objectName;
    private boolean closed = false;
    
    MinioOutputStream(MinioClient minioClient, String bucket, String objectName) {
        this.minioClient = minioClient;
        this.bucket = bucket;
        this.objectName = objectName;
    }
    
    @Override
    public void close() throws IOException {
        if (closed) {
            return;
        }
        
        try {
            byte[] data = toByteArray();
            ByteArrayInputStream inputStream = new ByteArrayInputStream(data);
            
            minioClient.putObject(PutObjectArgs.builder()
                .bucket(bucket)
                .object(objectName)
                .stream(inputStream, data.length, -1)
                .build());
                
            inputStream.close();
        } catch (Exception e) {
            throw new IOException("Failed to upload to MinIO: " + bucket + "/" + objectName, e);
        } finally {
            closed = true;
            super.close();
        }
    }
}