/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.management.minio.emf;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.impl.URIHandlerImpl;

import io.minio.GetObjectArgs;
import io.minio.MinioClient;
import io.minio.RemoveObjectArgs;
import io.minio.StatObjectArgs;

/**
 * MinIO URIHandler for EMF resources.
 * 
 * Handles URIs of the form: minio://alias/path/to/file.ecore
 * 
 * The alias in the URI authority maps to a specific MinIO configuration
 * that includes endpoint, credentials, and bucket information.
 */
public class MinioUriHandler extends URIHandlerImpl {
    
    private static final String SCHEME = "minio";
    
    private final MinioClient minioClient;
    private final String alias;
    private final String configuredBucket; // null if bucket should be extracted from URI
    
    /**
     * Creates a new MinIO URI handler.
     * 
     * @param minioClient the configured MinIO client
     * @param alias the alias this handler responds to
     * @param configuredBucket the bucket to use for operations, or null to extract from URI
     */
    public MinioUriHandler(MinioClient minioClient, String alias, String configuredBucket) {
        this.minioClient = minioClient;
        this.alias = alias;
        this.configuredBucket = configuredBucket;
    }
    
    @Override
    public boolean canHandle(URI uri) {
        return uri.scheme() != null && uri.scheme().equalsIgnoreCase(SCHEME)
            && alias.equals(extractAlias(uri));
    }
    
    @Override
    public InputStream createInputStream(URI uri, Map<?, ?> options) throws IOException {
        String bucket = resolveBucket(uri);
        String objectName = resolveObjectName(uri);
        
        try {
            return minioClient.getObject(GetObjectArgs.builder()
                .bucket(bucket)
                .object(objectName)
                .build());
        } catch (Exception e) {
            throw new IOException("Failed to create input stream for MinIO object: " + uri, e);
        }
    }
    
    @Override
    public OutputStream createOutputStream(URI uri, Map<?, ?> options) throws IOException {
        String bucket = resolveBucket(uri);
        String objectName = resolveObjectName(uri);
        
        return new MinioOutputStream(minioClient, bucket, objectName);
    }
    
    @Override
    public void delete(URI uri, Map<?, ?> options) throws IOException {
        String bucket = resolveBucket(uri);
        String objectName = resolveObjectName(uri);
        
        try {
            minioClient.removeObject(RemoveObjectArgs.builder()
                .bucket(bucket)
                .object(objectName)
                .build());
        } catch (Exception e) {
            throw new IOException("Failed to delete MinIO object: " + uri, e);
        }
    }
    
    @Override
    public boolean exists(URI uri, Map<?, ?> options) {
        String bucket = resolveBucket(uri);
        String objectName = resolveObjectName(uri);
        
        try {
            minioClient.statObject(StatObjectArgs.builder()
                .bucket(bucket)
                .object(objectName)
                .build());
            return true;
        } catch (Exception e) {
            return false;
        }
    }
    
    /**
     * Extracts the alias from the URI authority.
     * For URI "minio://myalias/folder/file.ecore", returns "myalias"
     */
    private String extractAlias(URI uri) {
        String authority = uri.authority();
        if (authority == null || authority.isEmpty()) {
            throw new IllegalArgumentException("MinIO URI must contain alias in authority: " + uri);
        }
        return authority;
    }
    
    /**
     * Resolves the bucket name based on configuration and URI.
     * If bucket is configured, uses that. Otherwise extracts from first URI path segment.
     */
    private String resolveBucket(URI uri) {
        if (configuredBucket != null && !configuredBucket.trim().isEmpty()) {
            return configuredBucket;
        }
        
        // Extract bucket from first path segment
        String path = uri.path();
        if (path == null || path.isEmpty() || path.equals("/")) {
            throw new IllegalArgumentException("MinIO URI must contain bucket in path when not configured: " + uri);
        }
        
        // Remove leading slash and get first segment
        String pathWithoutSlash = path.startsWith("/") ? path.substring(1) : path;
        int firstSlash = pathWithoutSlash.indexOf('/');
        
        if (firstSlash == -1) {
            throw new IllegalArgumentException("MinIO URI must contain object path after bucket: " + uri);
        }
        
        return pathWithoutSlash.substring(0, firstSlash);
    }
    
    /**
     * Resolves the object name based on configuration and URI.
     * If bucket is configured, entire path is object name. Otherwise path after first segment.
     */
    private String resolveObjectName(URI uri) {
        String path = uri.path();
        if (path == null) {
            throw new IllegalArgumentException("MinIO URI must contain object path: " + uri);
        }
        
        String pathWithoutSlash = path.startsWith("/") ? path.substring(1) : path;
        
        if (configuredBucket != null && !configuredBucket.trim().isEmpty()) {
            // Bucket is configured, entire path is object name
            return pathWithoutSlash;
        } else {
            // Bucket from path, object name is everything after first segment
            int firstSlash = pathWithoutSlash.indexOf('/');
            if (firstSlash == -1) {
                throw new IllegalArgumentException("MinIO URI must contain object path after bucket: " + uri);
            }
            return pathWithoutSlash.substring(firstSlash + 1);
        }
    }
}