/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.management.minio.emf;

import org.eclipse.emf.ecore.resource.URIHandler;
import org.gecko.emf.osgi.UriHandlerProvider;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.metatype.annotations.AttributeDefinition;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import io.minio.MinioClient;

/**
 * OSGi component that provides MinIO URIHandler for EMF resources.
 * 
 * This component requires configuration via OSGi Configuration Admin.
 * Multiple instances can be created with different configuration PIDs
 * to support different MinIO endpoints.
 * 
 * Configuration properties:
 * - endpoint: MinIO server endpoint (e.g., "http://localhost:9000")
 * - accessKey: MinIO access key
 * - secretKey: MinIO secret key
 * - region: MinIO region (optional, defaults to "us-east-1")
 */
@Component(
    service = UriHandlerProvider.class,
    configurationPolicy = ConfigurationPolicy.REQUIRE,
    immediate = true
)
@Designate(ocd = MinioUriHandlerProvider.Config.class)
public class MinioUriHandlerProvider implements UriHandlerProvider {
    
    @ObjectClassDefinition(
        name = "MinIO EMF URI Handler Configuration",
        description = "Configuration for MinIO URI handler that enables EMF resources to be stored in MinIO"
    )
    public @interface Config {
    	
    	@AttributeDefinition(
    			name = "Alias",
    			description = "The host alias, to not expose the condifugration to the URI"
    	)
    	String alias();
        
        @AttributeDefinition(
            name = "Endpoint",
            description = "MinIO server endpoint URL (e.g., http://localhost:9000)"
        )
        String endpoint();
        
        @AttributeDefinition(
            name = "Access Key",
            description = "MinIO access key for authentication"
        )
        String accessKey();
        
        @AttributeDefinition(
            name = "Secret Key",
            description = "MinIO secret key for authentication"
        )
        String secretKey();
        
        @AttributeDefinition(
            name = "Region",
            description = "MinIO region (optional, defaults to us-east-1)"
        )
        String region() default "us-east-1";
        
        @AttributeDefinition(
            name = "Bucket",
            description = "MinIO bucket name (optional, if not set bucket will be extracted from URI path)"
        )
        String bucket() default "";
    }
    
    private MinioClient minioClient;
    private MinioUriHandler uriHandler;
    
    @Activate
    void activate(Config config) {
        this.minioClient = MinioClient.builder()
            .endpoint(config.endpoint())
            .credentials(config.accessKey(), config.secretKey())
            .region(config.region())
            .build();
            
        String bucket = config.bucket().trim().isEmpty() ? null : config.bucket();
        this.uriHandler = new MinioUriHandler(minioClient, config.alias(), bucket);
    }
    
    @Deactivate
    void deactivate() {
        this.uriHandler = null;
        this.minioClient = null;
    }
    
    @Override
    public URIHandler getURIHandler() {
        return uriHandler;
    }
}