/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.management.minio.impl;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.mac.management.minio.storage.MinioStorageHelper;
import org.gecko.mac.mgmt.annotations.MacCapabilityConstants;
import org.gecko.mac.mgmt.api.EObjectRegistryService;
import org.gecko.mac.mgmt.api.EObjectStorageService;
import org.gecko.mac.mgmt.management.StorageBackendType;
import org.gecko.mac.mgmt.storage.AbstractEObjectStorageService;
import org.gecko.mac.mgmt.storage.AbstractStorageHelper;
import org.osgi.annotation.bundle.Capability;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.propertytypes.ServiceDescription;
import org.osgi.service.metatype.annotations.AttributeDefinition;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

/**
* MinIO implementation of EObjectStorageService using MinioStorageHelper.
* 
* This service uses the MinioStorageHelper internally which leverages:
* - AbstractStorageHelper for EMF operations
* - MinIO URIHandler for transport operations
* 
* URIs use the format: minio://alias/[pathPrefix/]objectId.ext
* 
* @author Mark Hoffmann
*/
@Component(
	service = EObjectStorageService.class,
    property = "storage.backend=minio",
    configurationPid = EObjectMinioStorageService.PID,
    configurationPolicy = ConfigurationPolicy.REQUIRE
)
@Designate(ocd = EObjectMinioStorageService.Config.class)
@Capability(namespace = MacCapabilityConstants.NAMESPACE_MAC_MANAGEMENT, name = MacCapabilityConstants.CAP_EOBJECT_STORAGE, version = "1.0", attribute = "storage.backend=minio")
@ServiceDescription("MinIO-based storage implementation for EObject workflow management")
public class EObjectMinioStorageService extends AbstractEObjectStorageService {

    @ObjectClassDefinition(
        name = "Simplified MinIO Storage Configuration",
        description = "Configuration for MinIO-based storage using AbstractEObjectStorageService"
    )
    public @interface Config {
        
        @AttributeDefinition(name = "MinIO Alias", description = "MinIO alias for URIHandler")
        String minioAlias();
        
        @AttributeDefinition(name = "Path Prefix", description = "Optional path prefix")
        String pathPrefix() default "";
        
        @AttributeDefinition(name = "Endpoint", description = "MinIO server endpoint")
        String endpoint() default "http://localhost:9000";
        
        @AttributeDefinition(name = "Access Key", description = "MinIO access key")
        String accessKey() default "minioadmin";
        
        @AttributeDefinition(name = "Secret Key", description = "MinIO secret key")
        String secretKey() default "minioadmin";
        
        @AttributeDefinition(name = "Bucket Name", description = "MinIO bucket name")
        String bucketName() default "";
        
        @AttributeDefinition(name = "Region", description = "MinIO region")
        String region() default "us-east-1";
        
        @AttributeDefinition(
            name = "Storage Role",
            description = "Role of this storage service (draft, approved, documentation, etc.)"
        )
        String storage_role() default "draft";
    }
    
    public static final String PID = "MinioEObjectStorage";
    private Config config;
    private String storageRole;
    
    @Reference(target = "(emf.name=management)")
    private ResourceSet resourceSet;
    
    @Reference
    private EObjectRegistryService<EObject> registry;
    
    @Activate
    public void activate(Config config) throws Exception {
        this.config = config;
        this.storageRole = config.storage_role();
        activateStorageService();
    }
    
    @Deactivate
    public void deactivate() {
        deactivateStorageService();
    }

    @Override
    protected AbstractStorageHelper createStorageHelper() throws Exception {
            
        // This would be the existing MinioStorageHelper
        return new MinioStorageHelper(resourceSet, config);
    }

    @Override
    public StorageBackendType getBackendType() {
        return StorageBackendType.MINIO;
    }

    /* 
     * (non-Javadoc)
     * @see org.gecko.mac.mgmt.storage.AbstractEObjectStorageService#getStorageRole()
     */
    @Override
    protected String getStorageRole() {
        return storageRole;
    }

    /* 
     * (non-Javadoc)
     * @see org.gecko.mac.mgmt.storage.AbstractEObjectStorageService#getRegistryService()
     */
    @Override
    protected EObjectRegistryService<EObject> getRegistryService() {
        return registry;
    }

}
