/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.management.minio.storage;

import java.io.IOException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.gecko.mac.management.minio.impl.EObjectMinioStorageService.Config;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.storage.AbstractStorageHelper;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

import io.minio.CopyObjectArgs;
import io.minio.CopySource;
import io.minio.Directive;
import io.minio.ListObjectsArgs;
import io.minio.MinioClient;
import io.minio.Result;
import io.minio.StatObjectArgs;
import io.minio.StatObjectResponse;
import io.minio.messages.Item;

/**
 * MinIO implementation of storage helper using EMF URIHandler approach with native metadata storage.
 * 
 * This helper leverages the MinIO URIHandler for all transport operations,
 * while AbstractStorageHelper handles EMF resource management and serialization.
 * Object metadata is stored natively using MinIO object metadata headers instead
 * of separate .metadata.xmi files, providing better performance and atomicity.
 * 
 * URIs use the format: minio://alias/[pathPrefix/]objectId.ext
 * 
 * @author Mark Hoffmann
 */
public class MinioStorageHelper extends AbstractStorageHelper {
    
    private static final Logger logger = Logger.getLogger(MinioStorageHelper.class.getName());
    
    // MinIO metadata header prefixes
    private static final String METADATA_PREFIX = "x-amz-meta-";
    private static final String UPLOAD_USER = METADATA_PREFIX + "upload-user";
    private static final String UPLOAD_TIME = METADATA_PREFIX + "upload-time";
    private static final String SOURCE_CHANNEL = METADATA_PREFIX + "source-channel";
    private static final String CONTENT_HASH = METADATA_PREFIX + "content-hash";
    private static final String OBJECT_TYPE = METADATA_PREFIX + "object-type";
    private static final String REVIEW_USER = METADATA_PREFIX + "review-user";
    private static final String REVIEW_TIME = METADATA_PREFIX + "review-time";
    private static final String REVIEW_REASON = METADATA_PREFIX + "review-reason";
    private static final String PROPERTIES = METADATA_PREFIX + "properties";
    
    private final String minioAlias;
    private final String pathPrefix;
    private final MinioClient minioClient;
    private final String bucketName;
    private final ObjectMapper objectMapper;

    public MinioStorageHelper(ResourceSet resourceSet, Config config) {
    	super(resourceSet);
    	this.minioAlias = config.minioAlias();
    	this.pathPrefix = config.pathPrefix() != null ? config.pathPrefix().trim() : "";
    	this.bucketName = config.bucketName();
    	this.minioClient = createMinioClient(config);
    	this.objectMapper = new ObjectMapper();
    }
    
    /**
     * Create MinIO client for listing operations
	 * @return the {@link MinioClient}
	 */
	private MinioClient createMinioClient(Config config) {
        return MinioClient.builder()
            .endpoint(config.endpoint())
            .credentials(config.accessKey(), config.secretKey())
            .region(config.region())
            .build();
	}

    @Override
    protected URI createStorageURI(String path) {
        // Create MinIO URI that will be handled by MinIO URIHandler
        String fullPath = pathPrefix.isEmpty() ? path : pathPrefix + "/" + path;
        return URI.createURI("minio://" + minioAlias + "/" + fullPath);
    }

    @Override
    protected void persistResource(String path, Resource resource) throws IOException {
        // No additional persistence needed - URIHandler handles everything
        // This method is called after resource.save() which already persisted via URIHandler
    }
    
    @Override
    public void saveMetadata(String objectId, ObjectMetadata metadata) throws IOException {
        // With native metadata storage, metadata is attached to the object itself
        // We override this to be a no-op since metadata is handled during object save
        // The metadata will be stored when saveObject() is called via attachMetadata()
    }
    
    @Override
    public ObjectMetadata loadMetadata(String objectId) throws IOException {
        String objectPath = findObjectPath(objectId);
        if (objectPath == null) {
            return null;
        }
        
        try {
            URI objectUri = createStorageURI(objectPath);
            String objectName = extractObjectNameFromUri(objectUri);
            
            StatObjectResponse stat = minioClient.statObject(
                StatObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .build()
            );
            
            return createMetadataFromHeaders(stat.userMetadata());
            
        } catch (Exception e) {
            throw new IOException("Failed to load metadata for object: " + objectId, e);
        }
    }
    
    @Override
    public void saveEObject(String objectId, EObject object, ObjectMetadata metadata) throws IOException {
        // First save the object normally using parent implementation
        super.saveEObject(objectId, object, metadata);
        
        // Then attach metadata to the MinIO object
        if (metadata != null) {
            attachMetadataToObject(objectId, metadata);
        }
    }

    @Override
    protected boolean storageExists(String path) throws IOException {
        URI uri = createStorageURI(path);
        URIConverter uriConverter = resourceSet.getURIConverter();
        return uriConverter.exists(uri, Collections.emptyMap());
    }

    @Override
    protected String findObjectPath(String objectId) throws IOException {
        // Try common extensions to find the object
        String[] extensions = {DEFAULT_EXTENSION, ".ecore", ".json"};
        
        for (String extension : extensions) {
            String path = buildObjectPath(objectId, extension);
            if (storageExists(path)) {
                return path;
            }
        }
        
        return null; // Object not found
    }

    @Override
    public boolean deleteObject(String objectId) throws IOException {
        URIConverter uriConverter = resourceSet.getURIConverter();
        
        // Delete object file (metadata is automatically deleted as it's stored with the object)
        String objectPath = findObjectPath(objectId);
        if (objectPath != null) {
            URI objectUri = createStorageURI(objectPath);
            uriConverter.delete(objectUri, Collections.emptyMap());
            return true;
        }
        
        return false;
    }

    @Override
    public List<String> listObjectIds() throws IOException {
        List<String> objectIds = new ArrayList<>();
        
        try {
            // Use MinIO client for listing (URIHandler doesn't provide listing capability)
            String listPrefix = pathPrefix.isEmpty() ? "" : pathPrefix + "/";
            
            Iterable<Result<Item>> results = minioClient.listObjects(
                ListObjectsArgs.builder()
                    .bucket(bucketName)
                    .prefix(listPrefix)
                    .build()
            );
            
            for (Result<Item> result : results) {
                Item item = result.get();
                String objectName = item.objectName();
                
                // Extract object ID from file names, avoiding metadata files
                String objectId = extractObjectIdFromPath(objectName);
                if (objectId != null && !objectIds.contains(objectId)) {
                    objectIds.add(objectId);
                }
            }
            
        } catch (Exception e) {
            throw new IOException("Failed to list objects from MinIO", e);
        }
        
        return objectIds;
    }
    
    /**
     * Extracts object ID from MinIO object path, avoiding metadata files.
     */
    private String extractObjectIdFromPath(String objectPath) {
        // Remove path prefix if present
        String fileName = objectPath;
        if (!pathPrefix.isEmpty() && objectPath.startsWith(pathPrefix + "/")) {
            fileName = objectPath.substring(pathPrefix.length() + 1);
        }
        
        // Skip metadata files - we only want to return each object ID once
        if (fileName.endsWith(METADATA_EXTENSION)) {
            return null;
        }
        
        // Extract object ID by removing known extensions
        if (fileName.endsWith(DEFAULT_EXTENSION)) {
            return fileName.substring(0, fileName.length() - DEFAULT_EXTENSION.length());
        } else if (fileName.endsWith(".ecore")) {
            return fileName.substring(0, fileName.length() - ".ecore".length());
        } else if (fileName.endsWith(".json")) {
            return fileName.substring(0, fileName.length() - ".json".length());
        }
        
        // For other extensions, remove everything after last dot
        int lastDot = fileName.lastIndexOf('.');
        if (lastDot > 0) {
            return fileName.substring(0, lastDot);
        }
        
        return fileName; // No extension found
    }
    
    /**
     * Extracts the object name from a MinIO URI for use with MinIO client operations.
     */
    private String extractObjectNameFromUri(URI uri) {
        String path = uri.path();
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        return path;
    }
    
    /**
     * Attaches metadata to an existing MinIO object by copying it with updated metadata headers.
     */
    private void attachMetadataToObject(String objectId, ObjectMetadata metadata) throws IOException {
        String objectPath = findObjectPath(objectId);
        if (objectPath == null) {
            throw new IOException("Object not found: " + objectId);
        }
        
        try {
            URI objectUri = createStorageURI(objectPath);
            String objectName = extractObjectNameFromUri(objectUri);
            
            // Convert metadata to headers
            Map<String, String> headerMap = createHeadersFromMetadata(metadata);
            
            // Convert Map to Multimap for MinIO API
            Multimap<String, String> userMetadata = HashMultimap.create();
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                userMetadata.put(entry.getKey(), entry.getValue());
            }
            
            // Copy object with new metadata (MinIO doesn't support metadata-only updates)
            minioClient.copyObject(
                CopyObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .source(
                        CopySource.builder()
                            .bucket(bucketName)
                            .object(objectName)
                            .build()
                    )
                    .metadataDirective(Directive.REPLACE)
                    .userMetadata(userMetadata)
                    .build()
            );
            
        } catch (Exception e) {
            throw new IOException("Failed to attach metadata to object: " + objectId, e);
        }
    }
    
    /**
     * Converts ObjectMetadata to MinIO user metadata headers.
     */
    private Map<String, String> createHeadersFromMetadata(ObjectMetadata metadata) {
        Map<String, String> headers = new HashMap<>();
        
        if (metadata.getUploadUser() != null) {
            headers.put(UPLOAD_USER, metadata.getUploadUser());
        }
        if (metadata.getUploadTime() != null) {
            headers.put(UPLOAD_TIME, DateTimeFormatter.ISO_INSTANT.format(metadata.getUploadTime()));
        }
        if (metadata.getSourceChannel() != null) {
            headers.put(SOURCE_CHANNEL, metadata.getSourceChannel());
        }
        if (metadata.getContentHash() != null) {
            headers.put(CONTENT_HASH, metadata.getContentHash());
        }
        if (metadata.getObjectType() != null) {
            headers.put(OBJECT_TYPE, metadata.getObjectType());
        }
        if (metadata.getReviewUser() != null) {
            headers.put(REVIEW_USER, metadata.getReviewUser());
        }
        if (metadata.getReviewTime() != null) {
            headers.put(REVIEW_TIME, DateTimeFormatter.ISO_INSTANT.format(metadata.getReviewTime()));
        }
        if (metadata.getReviewReason() != null) {
            headers.put(REVIEW_REASON, metadata.getReviewReason());
        }
        
        // Serialize properties as JSON string using Jackson
        if (metadata.getProperties() != null && !metadata.getProperties().isEmpty()) {
            try {
                String propertiesJson = objectMapper.writeValueAsString(metadata.getProperties());
                headers.put(PROPERTIES, propertiesJson);
            } catch (Exception e) {
                // Log error but don't fail the entire operation
                logger.warning("Failed to serialize properties: " + e.getMessage());
            }
        }
        
        return headers;
    }
    
    /**
     * Creates ObjectMetadata from MinIO user metadata headers.
     */
    private ObjectMetadata createMetadataFromHeaders(Map<String, String> headers) {
        ObjectMetadata metadata = ManagementFactory.eINSTANCE.createObjectMetadata();
        
        metadata.setUploadUser(headers.get(UPLOAD_USER));
        metadata.setSourceChannel(headers.get(SOURCE_CHANNEL));
        metadata.setContentHash(headers.get(CONTENT_HASH));
        metadata.setObjectType(headers.get(OBJECT_TYPE));
        metadata.setReviewUser(headers.get(REVIEW_USER));
        metadata.setReviewReason(headers.get(REVIEW_REASON));
        
        // Parse time fields
        String uploadTimeStr = headers.get(UPLOAD_TIME);
        if (uploadTimeStr != null) {
            try {
                metadata.setUploadTime(Instant.parse(uploadTimeStr));
            } catch (Exception e) {
                // Ignore invalid timestamp
            }
        }
        
        String reviewTimeStr = headers.get(REVIEW_TIME);
        if (reviewTimeStr != null) {
            try {
                metadata.setReviewTime(Instant.parse(reviewTimeStr));
            } catch (Exception e) {
                // Ignore invalid timestamp
            }
        }
        
        // Deserialize properties from JSON string using Jackson
        String propertiesStr = headers.get(PROPERTIES);
        if (propertiesStr != null && !propertiesStr.trim().isEmpty()) {
            try {
                Map<String, String> properties = objectMapper.readValue(
                    propertiesStr, 
                    new TypeReference<Map<String, String>>() {}
                );
                if (properties != null) {
                    metadata.getProperties().putAll(properties);
                }
            } catch (Exception e) {
                // Log error but don't fail the entire operation
                logger.warning("Failed to deserialize properties: " + e.getMessage());
            }
        }
        
        return metadata;
    }
    
}