/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.mqtt.handler.impl;

import static java.util.Objects.isNull;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.fennec.codec.options.CodecOptionsBuilder;
import org.eclipse.fennec.codec.options.CodecOptionsBuilder.ClassOptionsBuilder;

/**
 * Helper class to parse codec options configuration from OSGi properties and
 * build EMF load options map.
 *
 * Configuration properties:
 * - sthbnd.mapping.codec.rootObject.uri - URI of root EClass (e.g., "https://eclipse.org/fennec/lorawan#//UplinkMessage")
 * - sthbnd.mapping.codec.useExtendedMetadata - Boolean flag for using extended metadata names
 * - sthbnd.mapping.codec.class.uri - URI of the class for class-specific configuration
 * - sthbnd.mapping.codec.typeStrategy - Type strategy (e.g., "URI", "NAME")
 * - sthbnd.mapping.codec.typeMap.<key> - Type mapping entries (e.g., "Dragino" -> "https://eclipse.org/fennec/lorawan/dragino#//DraginoLSE01Uplink")
 * - sthbnd.mapping.codec.typeMapStrategy - Type map strategy (e.g., "MERGE", "REPLACE")
 *
 * @author Mark Hoffmann
 * @since 17.10.2025
 */
public class CodecOptionsConfigHelper {

	private static final String CODEC_PREFIX = "codec.";
	private static final String ROOT_OBJECT_URI = CODEC_PREFIX + "rootObject.uri";
	private static final String USE_EXTENDED_METADATA = CODEC_PREFIX + "useExtendedMetadata";
	private static final String CLASS_URI = CODEC_PREFIX + "class.uri";
	private static final String TYPE_STRATEGY = CODEC_PREFIX + "typeStrategy";
	private static final String TYPE_MAP_PREFIX = CODEC_PREFIX + "typeMap.";
	private static final String TYPE_MAP_STRATEGY = CODEC_PREFIX + "typeMapStrategy";

	private final ResourceSet resourceSet;
	private final Map<String, Object> properties;

	/**
	 * Constructor
	 *
	 * @param resourceSet the ResourceSet for resolving EClass URIs
	 * @param properties the configuration properties
	 */
	public CodecOptionsConfigHelper(ResourceSet resourceSet, Map<String, Object> properties) {
		this.resourceSet = resourceSet;
		this.properties = properties;
	}

	/**
	 * Builds the load options map from the configuration properties.
	 *
	 * @return Map of EMF load options
	 * @throws IllegalArgumentException if configuration is invalid
	 */
	public Map<String, Object> buildLoadOptions() {
		CodecOptionsBuilder optionsBuilder = CodecOptionsBuilder.create();

		// Root object configuration
		String rootObjectUri = getStringProperty(ROOT_OBJECT_URI);
		if (!isNull(rootObjectUri)) {
			EClass rootObject = resolveEClass(rootObjectUri);
			optionsBuilder.rootObject(rootObject);
		}

		// Use extended metadata
		Boolean useExtendedMetadata = getBooleanProperty(USE_EXTENDED_METADATA);
		if (!isNull(useExtendedMetadata)) {
			optionsBuilder.useNamesFromExtendedMetadata(useExtendedMetadata);
		}

		// Class-specific configuration
		String classUri = getStringProperty(CLASS_URI);
		if (!isNull(classUri)) {
			EClass eClass = resolveEClass(classUri);
			
			ClassOptionsBuilder classOptionsBuilder = optionsBuilder.forClass(eClass);
			// Type strategy
			String typeStrategy = getStringProperty(TYPE_STRATEGY);
			if (!isNull(typeStrategy)) {
				classOptionsBuilder.typeStrategy(typeStrategy);
			}

			// Type map
			Map<String, String> typeMap = getTypeMap();
			if (!typeMap.isEmpty()) {
				classOptionsBuilder.typeMap(typeMap);
			}

			// Type map strategy
			String typeMapStrategy = getStringProperty(TYPE_MAP_STRATEGY);
			if (!isNull(typeMapStrategy)) {
				classOptionsBuilder.typeMapStrategy(typeMapStrategy);
			}
		}
		return optionsBuilder.build();
	}

	/**
	 * Resolves an EClass from its URI fragment.
	 *
	 * @param uri the URI fragment (e.g., "https://eclipse.org/fennec/lorawan#//UplinkMessage")
	 * @return the resolved EClass
	 * @throws IllegalArgumentException if the URI cannot be resolved or is not an EClass
	 */
	private EClass resolveEClass(String uri) {
		if (isNull(uri) || uri.isEmpty()) {
			throw new IllegalArgumentException("EClass URI cannot be null or empty");
		}
		// Get the EPackage
		EObject eObject = resourceSet.getEObject(URI.createURI(uri), false);
		
		if (isNull(eObject)) {
			throw new IllegalArgumentException("EObject not found for URI: " + uri);
		}
		if(eObject instanceof EClass eClass) {
			return eClass;
		} else {
			throw new IllegalArgumentException(String.format("EObject with URI %s is not of type EClass", uri));
		}
	}

	/**
	 * Extracts type map entries from configuration properties.
	 *
	 * @return Map of type mappings
	 */
	private Map<String, String> getTypeMap() {
		Map<String, String> typeMap = new HashMap<>();

		for (Map.Entry<String, Object> entry : properties.entrySet()) {
			String key = entry.getKey();
			if (key.startsWith(TYPE_MAP_PREFIX)) {
				String typeKey = key.substring(TYPE_MAP_PREFIX.length());
				String typeUri = String.valueOf(entry.getValue());
				typeMap.put(typeKey, typeUri);
			}
		}

		return typeMap;
	}

	/**
	 * Gets a string property value.
	 *
	 * @param key the property key
	 * @return the property value or null if not found
	 */
	private String getStringProperty(String key) {
		Object value = properties.get(key);
		return isNull(value) ? null : String.valueOf(value);
	}

	/**
	 * Gets a boolean property value.
	 *
	 * @param key the property key
	 * @return the property value or null if not found
	 */
	private Boolean getBooleanProperty(String key) {
		Object value = properties.get(key);
		if (isNull(value)) {
			return null;
		}
		if (value instanceof Boolean) {
			return (Boolean) value;
		}
		return Boolean.parseBoolean(String.valueOf(value));
	}
}
