/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.mqtt.handler.impl;

import static java.util.Objects.isNull;
import static java.util.Objects.requireNonNull;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.sensinact.core.command.AbstractSensinactCommand;
import org.eclipse.sensinact.core.command.GatewayThread;
import org.eclipse.sensinact.core.model.SensinactModelManager;
import org.eclipse.sensinact.core.twin.SensinactDigitalTwin;
import org.eclipse.sensinact.mapping.ProviderMapping;
import org.gecko.emf.sensinact.model.ProviderMappingRegistry;
import org.gecko.emf.sensinact.model.ValueMapper;
import org.gecko.emf.sensinact.model.ValueMapperFactory;
import org.gecko.mac.audit.ActionType;
import org.gecko.mac.audit.Audit;
import org.gecko.mac.audit.CategoryType;
import org.gecko.mac.audit.helper.AuditHelper;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.mac.mqtt.handler.SouthboundMappingService;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.typedevent.TypedEventBus;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.promise.Promises;

/**
 * Implementation of a configurable mapping service implementation. It deserializes the data fom an byte representation
 * and gets the mapping from the registry and maps the data directly into SensiNact.
 * 
 * @author Mark Hoffmann
 * @since 14.08.2025
 */
@Component(name = "SouthboundMappingService", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class ConfigurableSouthboundMappingService implements SouthboundMappingService {

	private static final Logger LOGGER = Logger.getLogger(ConfigurableSouthboundMappingService.class.getName());
	private static final String AUDIT_TOPIC = "audit";

	@Reference
	private ProcessAuditSessionManager sessionManager;
	@Reference
	private ResourceSet resourceSet;
	@Reference
	private GatewayThread gatewayThread;
	@Reference
	private ProviderMappingRegistry registry;
	@Reference
	TypedEventBus typedEventBus;

	private final PromiseFactory pf = new PromiseFactory(Executors.newWorkStealingPool());
	private Map<String, Object> loadOptions;

	@Activate
	public void activate(ComponentContext cctx) throws ConfigurationException {
		createLoadOptions(FrameworkUtil.asMap(cctx.getProperties()));
	}

	@Modified
	public void modified(Map<String, Object> properties) {
		LOGGER.info("ConfigurableSouthboundMappingService config has been modified");
		createLoadOptions(properties);
	}

	@Deactivate
	public void deactivate() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.sensinact.southbound.emf.mapping.SouthboundMappingService#mapFrom
	 * (java.io.InputStream)
	 */
	@Override
	public Promise<EObject> mapFrom(InputStream data) {
		try {
			final EObject dataEObject = loadData(data);
			List<ProviderMapping> mappings = registry.getProviderMapping(dataEObject.eClass());
			if (mappings.size() != 1) {
				return pf.failed(new IllegalStateException("No mapping found for EClass '" + dataEObject.eClass().getName() + "'."));
			}
			ProviderMapping mapping = mappings.getFirst();
			ProcessAuditSession currentSession = retrieveCurrentSession();
			String sessionId = currentSession.getSessionId();
			gatewayThread.execute(new AbstractSensinactCommand<Boolean>() {
				@Override
				protected Promise<Boolean> call(SensinactDigitalTwin twin, SensinactModelManager modelManager,
						PromiseFactory pf) {
					ProcessAuditSession session = sessionManager.getSession(sessionId);
					sessionManager.setCurrentSession(session);
					Audit audit = null;
					try {						
						ValueMapper mapper = ValueMapperFactory.createValueMapper(twin, mapping);
						mapper.mapInstance(dataEObject);
						audit =  AuditHelper.createAudit(EcoreUtil.getIdentification(dataEObject), Instant.now().toString(), 
								"eclipse.sensinact", ActionType.MAPPED_TO_SENSINACT, CategoryType.SENSOR_DATA,
								String.format("Data has been propertly mapped to sensinact using mapping with id %s", mapping.getMid()));
						typedEventBus.deliver(AUDIT_TOPIC, audit);
						session.checkpoint(String.format("Data has been propertly mapped to sensinact using mapping with id %s", mapping.getMid()),
								audit);
						return pf.resolved(Boolean.TRUE);
					} catch (Exception e) {
						audit =  AuditHelper.createAudit(EcoreUtil.getIdentification(dataEObject), Instant.now().toString(), 
								"eclipse.sensinact", ActionType.FAILURE, CategoryType.SENSOR_DATA,
								String.format("Exception while mapping to sensinact using mapping with id %s", mapping.getMid()));
						typedEventBus.deliver(AUDIT_TOPIC, audit);
						session.checkpoint(String.format("Exception while mapping to sensinact using mapping with id %s", mapping.getMid()),
								audit);
						return pf.failed(e);
					}
				}
			}).getValue();
			return pf.resolved(dataEObject);
		} catch(IOException e) {
			return pf.failed(e);
		}
		catch (Exception e) {
			return pf.failed(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.sensinact.southbound.emf.mapping.SouthboundMappingService#mapFrom
	 * (byte[])
	 */
	@Override
	public Promise<EObject> mapFrom(byte[] data) {
		if (isNull(data)) {
			return Promises.failed(new NullPointerException("data parameter must not be null"));
		}
		if (data.length == 0) {
			return Promises.resolved(null);
		}
		return mapFrom(new ByteArrayInputStream(data));
	}

	private Resource createResource() {
		Resource resource;
		String path = UUID.randomUUID().toString();
		resource = resourceSet.createResource(URI.createURI(path + ".json"));
		requireNonNull(resource, String.format("Error creating resource for path: '%s'", path));
		return resource;
	}

	/**
	 * Create the load options, when using the right prefix
	 *
	 * @param properties the service properties
	 */
	private void createLoadOptions(Map<String, Object> properties) {
		if (isNull(properties)) {
			return;
		}

		// Extract simple mapping properties with prefix
		Map<String, Object> simpleMappingOptions = properties.entrySet().stream()
				.filter(e -> e.getKey().startsWith(PROP_PREFIX))
				.collect(Collectors.toMap(e -> e.getKey().replace(PROP_PREFIX, ""), Map.Entry::getValue));

		CodecOptionsConfigHelper codecHelper = new CodecOptionsConfigHelper(resourceSet, simpleMappingOptions);
		Map<String, Object> codecOptions = codecHelper.buildLoadOptions();

		loadOptions = new java.util.HashMap<>(codecOptions);

	}

	/**
	 * Loads the data from the input stream
	 * 
	 * @param data the input stream to load from
	 * @return an {@link EObject} or throws an exception
	 * @throws IOException
	 */
	private EObject loadData(InputStream data) throws IOException {
		requireNonNull(data, "The data parameter must not be null");
		Resource resource = createResource();
		resource.load(data, loadOptions);
		if (resource.getContents().isEmpty()) {
			throw new IllegalStateException("There has no content been loaded fromn the resource");
		}
		EObject content = resource.getContents().get(0);
		Audit audit = AuditHelper.createAudit(EcoreUtil.getIdentification(content), Instant.now().toString(), 
				"eclipse.fennec.emf.osgi", ActionType.LOADED, CategoryType.SENSOR_DATA,
				String.format("Data has been propertly loaded into EClass %s of model %s", content.eClass().getName(), content.eClass().getEPackage().getNsURI()));
		typedEventBus.deliver(AUDIT_TOPIC, audit);
		ProcessAuditSession currentSession = retrieveCurrentSession();
		currentSession.checkpoint(String.format("Data has been propertly loaded into EClass %s of model %s", content.eClass().getName(), content.eClass().getEPackage().getNsURI()),
				audit);

		return content;
	}

	private ProcessAuditSession retrieveCurrentSession() {
		ProcessAuditSession currentSession = sessionManager.getCurrentSession();
		if(currentSession != null) {
			return currentSession;
		} else {
			LOGGER.severe(String.format("SESSION IS NULL FOR AUDITING!"));
			throw new IllegalStateException(String.format("SESSION IS NULL FOR AUDITING!"));
		}
	}

}
