/*
 * Copyright (c) 2012 - 2024 Data In Motion and others.
 * All rights reserved. 
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */

package org.gecko.mac.mqtt.handler.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.System.Logger;
import java.lang.System.Logger.Level;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.mac.audit.ActionType;
import org.gecko.mac.audit.Audit;
import org.gecko.mac.audit.CategoryType;
import org.gecko.mac.audit.helper.AuditHelper;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.mac.lorawan.LoRaWANPackage;
import org.gecko.mac.mgmt.api.EObjectDiscoveryService;
import org.gecko.mac.mgmt.api.EObjectGenerationService;
import org.gecko.mac.mqtt.handler.SensorConfig;
import org.gecko.mac.mqtt.handler.SouthboundMappingService;
import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.MessagingService;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.typedevent.TypedEventBus;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushEvent.EventType;
import org.osgi.util.pushstream.PushStream;

@Designate(factory = true, ocd = SensorConfig.class)
@Component(configurationPid = "SensorConnector", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class SensorMqttConnector {

	private static final Logger logger = System.getLogger(SensorMqttConnector.class.getName());
	private static final String AUDIT_TOPIC = "audit";

	@Reference
	private ProcessAuditSessionManager sessionManager;
	@Reference()
	private ResourceSet resourceSet;
	@Reference
	private SouthboundMappingService mappingService;
	@Reference
	EObjectDiscoveryService<EObject> discoveryService;
	@Reference
	EObjectGenerationService<EObject> generationService;
	@Reference
	TypedEventBus typedEventBus;

	private PushStream<Message> subscription;
	private SensorConfig config;
	private ComponentContext componentContext;
	private ServiceReference<MessagingService> messagingServiceReference;
	private ScheduledExecutorService scheduler;

	@Activate
	public void activate(SensorConfig config, ComponentContext componentContext) {
		this.config = config;
		this.componentContext = componentContext;
		this.scheduler = Executors.newSingleThreadScheduledExecutor();

		startSubscription();

		// If no service found initially, retry periodically
		if (subscription == null) {
			scheduler.scheduleWithFixedDelay(this::retryStartSubscription, 5, 10, TimeUnit.SECONDS);
		}
	}

	@Deactivate
	private void deactivate() {
		if (scheduler != null) {
			scheduler.shutdownNow();
		}

		if (subscription != null) {
			subscription.close();
			subscription = null;
		}

		if (messagingServiceReference != null) {
			try {
				componentContext.getBundleContext().ungetService(messagingServiceReference);
			} catch (Exception e) {
				logger.log(Level.WARNING, "Error ungetting MessagingService", e);
			}
			messagingServiceReference = null;
		}
	}

	private void retryStartSubscription() {
		if (subscription == null) {
			startSubscription();
			if (subscription != null && scheduler != null) {
				scheduler.shutdown(); // Stop retrying once successful
			}
		}
	}

	/**
	 * Start subscription to a topic
	 */
	private void startSubscription() {
		if (config != null && subscription == null) {
			try {
				MessagingService messaging = getMessagingService();
				if (messaging != null) {
					subscription = messaging.subscribe(config.topic());
					subscription.forEachEvent(this::handle);
					logger.log(Level.INFO,
							"Sensinact Ecowitt Connector started with messaging service ID: {0}, topic: {1}",
							config.messagingServiceId(), config.topic());
				} else {
					logger.log(Level.DEBUG, "MessagingService with ID {0} not yet available, will retry...",
							config.messagingServiceId());
				}
			} catch (Exception e) {
				logger.log(Level.ERROR, "Error subscribing to MQTT topic {0}.\n{1}", config.topic(), e);
			}
		}
	}

	/**
	 * Get a new instance of the {@link MessagingService}
	 * @return the {@link MessagingService} instance
	 */
	private MessagingService getMessagingService() {
		try {
			BundleContext bundleContext = componentContext.getBundleContext();
			String filter = "(id=" + config.messagingServiceId() + ")";
			ServiceReference<?>[] references = bundleContext.getServiceReferences(MessagingService.class.getName(),
					filter);

			if (references != null && references.length > 0) {
				@SuppressWarnings("unchecked")
				ServiceReference<MessagingService> ref = (ServiceReference<MessagingService>) references[0];
				messagingServiceReference = ref;
				return bundleContext.getService(messagingServiceReference);
			}
		} catch (Exception e) {
			logger.log(Level.ERROR, "Error looking up MessagingService with ID: {0}.\n{1}", config.messagingServiceId(),
					e);
		}
		return null;
	}

	/**
	 * Handles a push event
	 * @param event the event to handle
	 * @return the 
	 */
	private long handle(PushEvent<? extends Message> event) {
		EventType type = event.getType();
		switch (type) {
		case CLOSE:
			logger.log(Level.INFO, "PushStream closed.");
			break;
		case DATA:
			onMessage(event.getData());
			break;
		case ERROR:
			event.getFailure().printStackTrace();
			break;
		default:
			break;
		}
		return 0;
	}

	/**
	 * Called with the message data
	 * @param message
	 */
	private void onMessage(Message message) {
		Audit audit = AuditHelper.createAudit(message.topic(), Instant.now().toString(), 
				"eclipse.sensinact.southbound.mqtt", ActionType.RECEIVED, CategoryType.SENSOR_DATA,
				String.format("Received sensor data over MQTT for topic %s", message.topic()));
		typedEventBus.deliver(AUDIT_TOPIC, audit);
		ProcessAuditSession startSession = sessionManager.startSession("Sensor Data");
		startSession.checkpoint(String.format("Received sensor data over MQTT for topic %s", message.topic()), 
				audit);
		try (ByteArrayInputStream bas = new ByteArrayInputStream(message.payload().array())) {
			mappingService.mapFrom(bas).onFailure(t -> {
				logger.log(Level.ERROR, "Error reading data. Trying to generate a new model for the data.", t);
				delegateToAIGenerator(message);
			}).onSuccess(s -> {
				logger.log(Level.INFO, "Succesfully processed data from topic: " + message.topic());
				sessionManager.getSession(startSession.getSessionId()).complete("Succesfully processed data from topic: " + message.topic());
			});
		} catch(IOException e) {
			logger.log(Level.ERROR, "IOException while reading data. Trying to generate a new model for the data.", e);
			delegateToAIGenerator(message);
		} finally {
		}
	}

	private void delegateToAIGenerator(Message message) {
		logger.log(Level.INFO, String.format("Trying to generate via AI model for unknown sensor arrived on topic %s", message.topic()));
		if(config.parentSensorPackageURI() != null) {
			EPackage parentEPackage = resourceSet.getPackageRegistry().getEPackage(config.parentSensorPackageURI());
			if(parentEPackage != null && parentEPackage instanceof LoRaWANPackage lorawanPackage) {
				generationService.requestGeneration(new String(message.payload().array(), StandardCharsets.UTF_8), "SensiNact", "sys_sensinact", null, lorawanPackage);
			} else {
				logger.log(Level.ERROR, String.format("EPackage with URI %s not found. Cannot continue with AI generation.", config.parentSensorPackageURI()));
			}
		} else {
			generationService.requestGeneration(new String(message.payload().array(), StandardCharsets.UTF_8), "SensiNact", "sys_sensinact", null);
		}
	}
	
	
}
