/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.order.demo;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Logger;

import org.gecko.mac.audit.AuditEventData;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.mac.order.OrderService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;

/**
 * Demo showing multi-threaded session management and thread handover patterns.
 * 
 * @author Mark Hoffmann
 * @since 26.10.2025
 */
@Component(immediate = true)
public class MultiThreadAuditDemo {
    
    private static final Logger logger = Logger.getLogger(MultiThreadAuditDemo.class.getName());
    
    @Reference
    private ProcessAuditSessionManager sessionManager;
    
    @Reference
    private OrderService orderService;
    
    private ExecutorService executorService;
    
    @Activate
    public void activate() {
        executorService = Executors.newFixedThreadPool(3);
        logger.info("Starting Multi-Thread Audit Demo...");
        
        // Run demo after a short delay
        executorService.execute(this::runMultiThreadDemo);
    }
    
    @Deactivate
    public void deactivate() {
        if (executorService != null) {
            executorService.shutdown();
        }
    }
    
    private void runMultiThreadDemo() {
        try {
            Thread.sleep(3000); // Wait for proxy to be ready
            
            logger.info("=== Demo 1: Thread Isolation (Current Behavior) ===");
            demonstrateThreadIsolation();
            
            Thread.sleep(1000);
            
            logger.info("=== Demo 2: Manual Thread Handover ===");
            demonstrateManualThreadHandover();
            
            Thread.sleep(1000);
            
            logger.info("=== Demo 3: Async Processing with Session Handover ===");
            demonstrateAsyncWithHandover();
            
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
    
    /**
     * Demonstrates that sessions are thread-local by default.
     */
    private void demonstrateThreadIsolation() {
        // Start session in main thread
        ProcessAuditSession session = sessionManager.startSession("Thread Isolation Demo");
        session.addContext("mainThread", Thread.currentThread().getName());
        session.checkpoint("main-thread-start", "Session started in main thread");
        
        // Call service in main thread
        String result1 = orderService.getOrderDetails("ISO-001");
        logger.info("Main thread result: " + result1);
        
        // Try to access session from another thread
        CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
            logger.info("Worker thread: " + Thread.currentThread().getName());
            
            // This will return null because sessions are thread-local
            ProcessAuditSession workerSession = sessionManager.getCurrentSession();
            logger.info("Worker session: " + (workerSession != null ? workerSession.getSessionId() : "null"));
            
            // Service call in worker thread - will not be associated with main session
            String result2 = orderService.getUser("ISO-001");
            logger.info("Worker thread result: " + result2);
            
        }, executorService);
        
        try {
            future.get(); // Wait for completion
        } catch (Exception e) {
            logger.warning("Future failed: " + e.getMessage());
        }
        
        session.checkpoint("main-thread-end", "Main thread completed");
        session.complete("Thread isolation demo completed");
        
        displaySessionSummary("Thread Isolation", session);
    }
    
    /**
     * Demonstrates manual thread handover using session ID.
     */
    private void demonstrateManualThreadHandover() {
        // Start session in main thread
        ProcessAuditSession session = sessionManager.startSession("Manual Handover Demo");
        session.addContext("originalThread", Thread.currentThread().getName());
        session.checkpoint("handover-start", "Preparing for thread handover");
        
        // Get session ID for handover
        String sessionId = session.getSessionId();
        
        // Hand over to worker thread
        CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
            logger.info("Worker thread: " + Thread.currentThread().getName());
            
            // Retrieve session by ID and set as current for this thread
            ProcessAuditSession handedOverSession = sessionManager.getSession(sessionId);
            if (handedOverSession != null) {
                sessionManager.setCurrentSession(handedOverSession);
                
                // Add context from worker thread
                handedOverSession.addContext("workerThread", Thread.currentThread().getName());
                handedOverSession.checkpoint("worker-start", "Session active in worker thread");
                
                // Service calls will now be tracked in the handed-over session
                String result = orderService.processOrder("HANDOVER-001");
                logger.info("Worker result: " + result);
                
                handedOverSession.checkpoint("worker-complete", "Worker thread processing complete");
                
                // Clear session from worker thread
                sessionManager.clearCurrentSession();
            } else {
                logger.warning("Failed to retrieve session for handover");
            }
        }, executorService);
        
        try {
            future.get(); // Wait for completion
        } catch (Exception e) {
            logger.warning("Handover future failed: " + e.getMessage());
        }
        
        // Back in main thread - session should still be available
        session.checkpoint("handover-end", "Thread handover completed");
        session.complete("Manual handover demo completed");
        
        displaySessionSummary("Manual Handover", session);
    }
    
    /**
     * Demonstrates a more complex async processing scenario with proper session handover.
     */
    private void demonstrateAsyncWithHandover() {
        ProcessAuditSession session = sessionManager.startSession("Async Processing Demo");
        session.addContext("orderId", "ASYNC-002");
        session.addContext("userId", "user456");
        session.checkpoint("async-start", "Starting async order processing");
        
        String sessionId = session.getSessionId();
        
        // Stage 1: Order validation (async)
        CompletableFuture<String> validationFuture = CompletableFuture.supplyAsync(() -> {
            // Set up session in validation thread
            ProcessAuditSession asyncSession = sessionManager.getSession(sessionId);
            sessionManager.setCurrentSession(asyncSession);
            
            asyncSession.addContext("validationThread", Thread.currentThread().getName());
            asyncSession.checkpoint("validation-start", "Order validation started");
            
            String details = orderService.getOrderDetails("ASYNC-002");
            asyncSession.checkpoint("validation-complete", "Order validation completed");
            
            sessionManager.clearCurrentSession();
            return details;
        }, executorService);
        
        // Stage 2: User processing (async, depends on validation)
        CompletableFuture<String> userFuture = validationFuture.thenApplyAsync(orderDetails -> {
            // Set up session in user processing thread
            ProcessAuditSession asyncSession = sessionManager.getSession(sessionId);
            sessionManager.setCurrentSession(asyncSession);
            
            asyncSession.addContext("userProcessingThread", Thread.currentThread().getName());
            asyncSession.checkpoint("user-processing-start", "User processing started");
            
            String user = orderService.getUser("ASYNC-002");
            asyncSession.checkpoint("user-processing-complete", "User processing completed");
            
            sessionManager.clearCurrentSession();
            return user;
        }, executorService);
        
        // Stage 3: Final processing (async, depends on user processing)
        CompletableFuture<String> finalFuture = userFuture.thenApplyAsync(user -> {
            // Set up session in final processing thread
            ProcessAuditSession asyncSession = sessionManager.getSession(sessionId);
            sessionManager.setCurrentSession(asyncSession);
            
            asyncSession.addContext("finalProcessingThread", Thread.currentThread().getName());
            asyncSession.checkpoint("final-processing-start", "Final processing started");
            
            String result = orderService.processOrder("ASYNC-002");
            asyncSession.checkpoint("final-processing-complete", "Final processing completed");
            
            sessionManager.clearCurrentSession();
            return result;
        }, executorService);
        
        try {
            // Wait for all async processing to complete
            String finalResult = finalFuture.get();
            logger.info("Final async result: " + finalResult);
            
            session.checkpoint("async-end", "All async processing completed");
            session.complete("Async processing demo completed successfully");
            
        } catch (Exception e) {
            session.error("async-error", "Async processing failed", e);
            session.fail("Async processing demo failed: " + e.getMessage());
        }
        
        displaySessionSummary("Async Processing", session);
    }
    
    private void displaySessionSummary(String demoName, ProcessAuditSession session) {
        logger.info("=== " + demoName + " Session Summary ===");
        logger.info("Session ID: " + session.getSessionId());
        logger.info("Process: " + session.getProcessName());
        logger.info("Active: " + session.isActive());
        logger.info("Context: " + session.getAllContext());
        logger.info("Events (" + session.getEvents().size() + "):");
        
        for (AuditEventData event : session.getEvents()) {
            String duration = event.getDurationNs() != null ? 
                            " (" + event.getDurationNs() + " ns)" : "";
            String threadInfo = " [thread=" + event.getThreadId() + "]";
            logger.info("  " + event.getEventType() + ": " + event.getCheckpoint() + duration + threadInfo);
        }
        logger.info("========================================");
    }
}