/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.order.demo;

import java.util.logging.Logger;

import org.gecko.mac.audit.AuditEventData;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.mac.order.OrderService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * Demo component showing manual/programmatic session management combined with proxy-based auditing.
 * 
 * @author Mark Hoffmann
 * @since 26.10.2025
 */
//@Component
public class ProcessAuditDemo {
    
    private static final Logger logger = Logger.getLogger(ProcessAuditDemo.class.getName());
    
    @Reference
    private ProcessAuditSessionManager sessionManager;
    
    @Reference
    private OrderService orderService; // This will be the audited proxy
    
    @Activate
    public void activate() {
        logger.info("Starting Process Audit Demo...");
        
        // Run demo after a short delay to ensure proxy is available
        new Thread(this::runDemo).start();
    }
    
    private void runDemo() {
        try {
            Thread.sleep(2000); // Wait for proxy to be ready
            demonstrateSessionBasedWorkflow();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
    
    /**
     * Demonstrates a complete session-based workflow combining manual session management
     * with automatic proxy-based method auditing.
     */
    private void demonstrateSessionBasedWorkflow() {
        // 1. Start a process audit session
        ProcessAuditSession session = sessionManager.startSession("Order Processing Workflow");
        logger.info("Started session: " + session.getSessionId());
        
        try {
            // 2. Add business context to the session
            session.addContext("userId", "user123");
            session.addContext("orderId", "ORDER-456");
            
            // 3. Manual checkpoint - process initiation
            session.checkpoint("workflow-started", "Order processing workflow initiated");
            
            // 4. Call audited service methods - these will be automatically tracked in the session
            String orderDetails = orderService.getOrderDetails("ORDER-456");
            logger.info("Order details: " + orderDetails);
            
            // 5. Another checkpoint
            session.checkpoint("details-retrieved", "Order details successfully retrieved");
            
            // 6. Get user information
            String user = orderService.getUser("ORDER-456");
            logger.info("User: " + user);
            
            // 7. Process the order
            String processResult = orderService.processOrder("ORDER-456");
            logger.info("Process result: " + processResult);
            
            // 8. Final checkpoint
            session.checkpoint("processing-complete", "Order processing completed successfully");
            
            // 9. Complete the session
            session.complete("Order workflow completed successfully");
            
            // 10. Display session summary
            displaySessionSummary(session);
            
        } catch (Exception e) {
            session.error("workflow-error", "Workflow failed", e);
            session.fail("Order processing workflow failed: " + e.getMessage());
            logger.warning("Workflow failed: " + e.getMessage());
        }
    }
    
    private void displaySessionSummary(ProcessAuditSession session) {
        logger.info("=== Session Summary ===");
        logger.info("Session ID: " + session.getSessionId());
        logger.info("Process: " + session.getProcessName());
        logger.info("Active: " + session.isActive());
        logger.info("Events (" + session.getEvents().size() + "):");
        
        for (AuditEventData event : session.getEvents()) {
            String duration = event.getDurationNs() != null ? 
                            " (" + event.getDurationNs() + " ns)" : "";
            logger.info("  " + event.getEventType() + ": " + event.getCheckpoint() + duration);
        }
    }
}