/*
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Data In Motion - initial API and implementation
 */
package org.gecko.mac.sensor.data.generator;

import java.time.Instant;
import java.util.Random;
import java.util.UUID;

import org.gecko.mac.lorawan.DeviceInfo;
import org.gecko.mac.lorawan.LoRaWANFactory;
import org.gecko.mac.moisturesensor.AtomLite;
import org.gecko.mac.moisturesensor.AtomLiteObject;
import org.gecko.mac.moisturesensor.M5CalibrationData;
import org.gecko.mac.moisturesensor.M5MoistureReading;
import org.gecko.mac.moisturesensor.M5MoistureSensorData;
import org.gecko.mac.moisturesensor.M5MoistureUplink;
import org.gecko.mac.moisturesensor.MoistureSensorFactory;

public class MoistureSensorGenerator {

	private static final String ATOM_LITE_PROFILE_NAME = "M5AtomLite";
	private static final String M5_PROFILE_NAME = "M5MoistureV1";
	
	private static MoistureSensorFactory moistureSensorFactory = MoistureSensorFactory.eINSTANCE;
	private static LoRaWANFactory lorawanFactory = LoRaWANFactory.eINSTANCE;
	
	public static M5MoistureUplink createM5SensorData() {
		M5MoistureUplink sensorData = moistureSensorFactory.createM5MoistureUplink();
		sensorData.setAdr(new Random().nextBoolean());
		sensorData.setConfirmed(new Random().nextBoolean());
		sensorData.setData(UUID.randomUUID().toString());
		sensorData.setDeduplicationId(UUID.randomUUID().toString());
		sensorData.setDevAddr(UUID.randomUUID().toString());
		sensorData.setDr(new Random().nextInt());
		sensorData.setFCnt(new Random().nextInt());
		sensorData.setFPort(new Random().nextInt());
		sensorData.setTime(Instant.now().toString());
		sensorData.setDeviceInfo(createDeviceInfo(M5_PROFILE_NAME));
		sensorData.setObject(createM5Object());		
		return sensorData;
	}
	
	public static AtomLite createAtomLiteSensorData(String anotherName) {
		AtomLite sensorData = moistureSensorFactory.createAtomLite();
		sensorData.setAdr(new Random().nextBoolean());
		sensorData.setConfirmed(new Random().nextBoolean());
		sensorData.setData(UUID.randomUUID().toString());
		sensorData.setDeduplicationId(UUID.randomUUID().toString());
		sensorData.setDevAddr(UUID.randomUUID().toString());
		sensorData.setDr(new Random().nextInt());
		sensorData.setFCnt(new Random().nextInt());
		sensorData.setFPort(new Random().nextInt());
		sensorData.setTime(Instant.now().toString());
		sensorData.setDeviceInfo(createDeviceInfo(anotherName == null ? ATOM_LITE_PROFILE_NAME : anotherName));
		sensorData.setObject(createAtomLiteObject());
		return sensorData;
	}
	
	private static M5MoistureSensorData createM5Object() {
		M5MoistureSensorData object = moistureSensorFactory.createM5MoistureSensorData();
		object.setTimestamp(new Random().nextDouble());
		M5MoistureReading reading = moistureSensorFactory.createM5MoistureReading();
		reading.setPercent(new Random().nextInt());
		reading.setRaw(new Random().nextInt());
		object.setMoisture(reading);
		M5CalibrationData calibration = moistureSensorFactory.createM5CalibrationData();
		calibration.setDry(new Random().nextInt());
		calibration.setWet(new Random().nextInt());
		object.setCal(calibration);
		return object;
	}
	
	private static AtomLiteObject createAtomLiteObject() {
		AtomLiteObject object = moistureSensorFactory.createAtomLiteObject();
		object.setCalDry(new Random().nextFloat());
		object.setCalWet(new Random().nextFloat());
		object.setMoisturePercent(new Random().nextFloat());
		object.setMoistureRaw(new Random().nextFloat());
		object.setTimestamp(Instant.now().toString());
		return object;
	}
	
	private static DeviceInfo createDeviceInfo(String deviceName) {
		DeviceInfo deviceInfo = lorawanFactory.createDeviceInfo();
		deviceInfo.setApplicationId(UUID.randomUUID().toString());
		deviceInfo.setApplicationName(UUID.randomUUID().toString());
		deviceInfo.setDevEui(UUID.randomUUID().toString());
		deviceInfo.setDeviceName(deviceName);
		deviceInfo.setDeviceProfileName(deviceName);
		deviceInfo.setDeviceProfileId(UUID.randomUUID().toString());		
		return deviceInfo;
	}

}
