/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.sensor.data.generator;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.fennec.codec.options.CodecModuleOptions;
import org.gecko.emf.osgi.constants.EMFNamespaces;
import org.gecko.mac.moisturesensor.AtomLite;
import org.gecko.mac.moisturesensor.M5MoistureUplink;
import org.gecko.osgi.messaging.MessagingService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;

/**
 * 
 * @author ilenia
 * @since Oct 16, 2025
 */
@Component(immediate = true, name = "MoistureSensorGeneratorComponent")
public class MoistureSensorGeneratorComponent implements Runnable {
	
	private static final Logger LOGGER = Logger.getLogger(MoistureSensorGeneratorComponent.class.getName());
	
	private static final String ATOM_LITE_PUBLISH_TOPIC = "airquality/sensor";
	private static final String M5_PUBLISH_TOPIC = "jena/watering/sensor";
	
	private static final Map<String, Object> SAVE_OPTIONS = Map.ofEntries(Map.entry(CodecModuleOptions.CODEC_MODULE_SERIALIZE_TYPE, false), 
			Map.entry(CodecModuleOptions.CODEC_MODULE_USE_NAMES_FROM_EXTENDED_METADATA, true)); 
	private MessagingService msgService;
	private ScheduledExecutorService executorService;
	private ResourceSet resourceSet;

	@Activate
	public MoistureSensorGeneratorComponent(@Reference(target = "(id=dim)") MessagingService msgService, 
			@Reference(target = "(" + EMFNamespaces.EMF_CONFIGURATOR_NAME + "=CodecJson)") ResourceSet resourceSet) {
		this.msgService = msgService;
		this.resourceSet = resourceSet;
		executorService = Executors.newSingleThreadScheduledExecutor();
		executorService.scheduleAtFixedRate(this::run, 1, 1, TimeUnit.MINUTES);
	}
	
	@Deactivate
	public void deactivate() {
		if(executorService != null) {
			executorService.shutdown();
			executorService = null;
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		
		AtomLite atomLite = MoistureSensorGenerator.createAtomLiteSensorData("M5AirQ");
		try {
			msgService.publish(ATOM_LITE_PUBLISH_TOPIC, createMsgContent(atomLite));
			LOGGER.info(String.format("Succesfully published AtomLite sensor data to %s", ATOM_LITE_PUBLISH_TOPIC));
		} catch (Exception e) {
			LOGGER.log(Level.SEVERE, String.format("Error publishing AtomLite sensor data to %s", ATOM_LITE_PUBLISH_TOPIC), e);
		}
		
//		M5MoistureUplink milesight = MoistureSensorGenerator.createM5SensorData();
//		try {
//			msgService.publish(M5_PUBLISH_TOPIC, createMsgContent(milesight));
//			LOGGER.info(String.format("Succesfully published M5 sensor data to %s", M5_PUBLISH_TOPIC));
//		} catch (Exception e) {
//			LOGGER.log(Level.SEVERE, String.format("Error publishing M5 sensor data to %s", M5_PUBLISH_TOPIC), e);
//		}
	}
	
	ByteBuffer createMsgContent(EObject eObject) {
		Resource resource = resourceSet.createResource(URI.createURI(UUID.randomUUID().toString()+".json"));
		resource.getContents().add(eObject);
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			resource.save(outputStream, SAVE_OPTIONS);
			return ByteBuffer.wrap(outputStream.toByteArray());
		} catch (IOException e) {
			LOGGER.log(Level.SEVERE, String.format("Error creating message content for moisture sensor"), e);
			throw new IllegalStateException(String.format("Error creating message content for moisture sensor"));
		}
	}
}
