/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.sensor.data.generator;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.osgi.messaging.MessagingService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;

/**
 * This is a component to send over MQTT a new sensor data. 
 * The first time the sensor data is received it should be treated as unknown and 
 * it should trigger the AI EPakcage generation
 * @author ilenia
 * @since Oct 16, 2025
 */
//@Component(immediate = true, name = "UnknownMoistureSensorGeneratorComponent")
public class UnknownMoistureSensorGeneratorComponent implements Runnable {
	
	private static final Logger LOGGER = Logger.getLogger(UnknownMoistureSensorGeneratorComponent.class.getName());
	
	private static final String ATOM_LITE_PUBLISH_TOPIC = "moisture/atomlite";	
	private MessagingService msgService;
	private ScheduledExecutorService executorService;
	
	private static final String UNKNOWN_SENSOR = """		
				
					
						
								
			{
			"adr": true,
			"confirmed": true,
			"data": "74e4b6f1-1005-4b23-b026-8c25cc1a4b30",
			"deduplicationId": "82d9f9f6-4f5e-43fd-8358-7a670f4208f8",
			"devAddr": "d0b76d18-ce8d-4d28-9685-9fa278519e07",
			"deviceInfo": {
				"applicationId": "55bf86a6-2df2-4697-b184-68e758246919",
				"applicationName": "26d8442f-4d01-471b-9cba-30f5a6bf3d32",
				"devEui": "75826669-222f-40ba-82e2-3660c3572c64",
				"deviceName": "UNKNOWN-SENSOR",
				"deviceProfileId": "c8611aa3-f99b-4815-8dce-bce50d785585",
				"deviceProfileName": "UNKNOWN-SENSOR"
				},
			"dr": 927274469,
			"fCnt": -463748128,
			"fPort": 1723862512,
			"object": {
				"calibration_dry": %f,
				"calibration_wet": %f,
				"humidity_in_percent": %f,
				"humidity_raw": %f,
				"timestamp": "%s"
				},
			"time": "%s"
			}			
			""";

	@Activate
	public UnknownMoistureSensorGeneratorComponent(@Reference MessagingService msgService) {
		this.msgService = msgService;
		executorService = Executors.newSingleThreadScheduledExecutor();
		executorService.scheduleAtFixedRate(this::run, 1, 1, TimeUnit.MINUTES);
	}
	
	@Deactivate
	public void deactivate() {
		if(executorService != null) {
			executorService.shutdown();
			executorService = null;
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		
		String unknownSensorData = String.format(UNKNOWN_SENSOR, 
				new Random().nextDouble(), 
				new Random().nextDouble(),
				new Random().nextDouble(),
				new Random().nextDouble(),
				Instant.now().toString(),
				Instant.now().toString()
				);
		LOGGER.fine("Generated unknown sensor data: " + unknownSensorData);
		try {
			msgService.publish(ATOM_LITE_PUBLISH_TOPIC, ByteBuffer.wrap(unknownSensorData.getBytes()));
			LOGGER.info(String.format("Succesfully published UNKNOWN sensor data to %s", ATOM_LITE_PUBLISH_TOPIC));
		} catch (Exception e) {
			LOGGER.log(Level.SEVERE, String.format("Error publishing UNKNOWN sensor data to %s", ATOM_LITE_PUBLISH_TOPIC), e);
		}
	}
}
